/*!************************************************************************
 * FILE :         clAudioSMEngine.cpp
 *
 * SW-COMPONENT:  Audio statemachine
 *
 * DESCRIPTION:
 *                STARTUP:
 *                StraightForward Case is:
 *                - Check in persistent DP for LSM and PSM
 *                  LSM: Last Selected Mode, i.e. the Entertainment Source that was
 *                  played before switching off the system
 *                  PSM: Previous Selected Mode, i.e. the Entertainment Source that was
 *                  played before the LSM
 *                - Check if LSM still available
 *                  If LSM available: start it
 *                  If LSM not available: Check if PSM still available
 *                     If PSM available: start it
 *                     If PSM not available: Start Radio.FM1
 *                Startup Details:
 *                - Before any communication to FC-Audio is initiated, wait for
 *                  AudioService to become available.
 *                - Before the availability of a media source can be checked, wait for
 *                  MediaService to become available and wait, until information about
 *                  removable mediasources is available. (was SD removed from slot while off?)
 *                - Prevent, that the XML framework begins to place API-Calls like:
 *                  Sound.GetCurrentAudioComponent()
 *                  AVDC.GetCurrentSource()
 *                  AVDC.Audio.GetAudioSourceState(...)
 *                  before the mentioned Services are available.
 *                - Before starting LSM, ensure, that media-removed info is evaluated on stack
 *                  Don't try to start e.g. LSM:SD and then immediately stop it, because is was removed
 *                  during off-state.
 *
 *                  DATAPOOL:
 *                  The following DP entries are relevant for Audio Source Switching
 *                  and the corresponding Screen-Changes / visual animations:
 *                  DPAUDIO__PER_ENTSRC:
 *                     Stores the current Audio Component (Radio or Media)
 *                     Used to determine/switch the menu net Radio vs. Media
 *                     Not persistent.
 *                  DPAUDIO__ENT_SRC:
 *                     Stores the current Radio-Band. (AM, FM, DAB)
 *                     Used to determine/switch the screen in Radio-Menu
 *                     In addition this value is required to maintain the history
 *                     Persistent.
 *                  DPAVDC__CURSRC:
 *                     Stores the current Media-Source (SD, CD-DA, CDC, etc)
 *                     Used to determine/switch the screen in Media-Menu
 *                     In addition this value is required to maintain the history
 *                     Persistent.
 *                  DPAVDC_PREV_AUDIO_SRC and DPAVDC_PCUR_AUDIO_SRC:
 *                     Stores the current and previous (Entertainment-)Audio-Source
 *                     Not used for Screen Changes. No GUI-Data Events.
 *                     Used for the history-stack for entertainment sources in the Audio-handler
 *                     Persistent.
 *
 * AUTHOR:        CM-DI/PJ-VW34 Steilmann
 *
 * COPYRIGHT:     (c) 2006 Blaupunkt Werke
 * HISTORY:
 *  19.09.07       CM-DI/PJ-VW34 Steilmann
 *  Initial Version
 *  23.10.07       CM-DI/PJ-VW34 Steinle
 *                 moved Creation of AudioSourceController from Contructor
 *                 to bConfigure.
 * 06.03.2008      CM-DI/PJ-VW34 Steinle
 *                 AudioEngine requests are allowed now also when AudioEngine is not started
 *                 This is posibble due to the Srv-Unavailable-handling in
 *                 the audio-src-implementations
 * 07.03.2008      CM-DI/PJ-VW34 Steinle
 *                 fix for mediaSourceRemoved-Event. If PSM=radio-am
 *                 goto am (not fm). Solved with isEntertainmentsourceAvailable()
 * 08.03.2008      CM-DI/PJ-VW34 Steinle
 *                 introduced persistence for mute-state and separate mute-pin source
 * 17.07.2013      CM-AI/CB-3 R.Volkmer (mailto:R.Volkmer@ITB-Solutions)
 *             initial extraction from HMI and integrate to Genivi AudioManager CtrlPlugIn
 *************************************************************************/

#include "AudioStack/clAudioSMEngine.h"
#include "AudioStack/SMT/clSrcStateFactory.h"
#include "AudioStack/AudioSources/clAudioSourceFactory.h"

#include "../../config/dp/AudioUserDpIfSelect.h"
#include<string>
//#define DP_DATAPOOL_ID =0xE880
#define DP_S_IMPORT_INTERFACE_FI
//#include "dp_if.h"

#ifndef VARIANT_S_FTR_ENABLE_AUDIO_STACK_UTEST
#include "dp_e880_if.h"
#else
#include "dp_e880_mock_if.h"
#endif
#define SINK_SIZE 20
#define OEM_TYPE_INFINITI     0x16
// Unorganised traces
#ifndef USE_DLT_TRACE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_COMP_AUDIOSTACK
#include "trcGenProj/Header/clAudioSMEngine.cpp.trc.h"
#endif
#endif

#define CFC_FI_S_IMPORT_INTERFACE_FI_TYPES
#include "cfc_fi_if.h"


#include "AudioStack/clTogglingSequences.h"
#include "AudioStack/clGeniviAudioCtrlAdapter.h"
#include "AudioStack/AudioSources/clFactory_AudioSourceClass.h"
namespace AudioStack
{

using namespace AudioSource;

tU8                        clAudioSMEngine::m_u8ErrorCode                     = AUDIOSM_OK;
tU8                        clAudioSMEngine::m_u8AudioRouteMgr_SrvState        = 0;//AMT_C_U8_SVCSTATE_NOT_AVAILABLE;
tU8                        clAudioSMEngine::m_u8Runlevel                      = 0;
SourceID                   clAudioSMEngine::m_enSource_Entertainment_Previous = SourceID(clFactory_AudioSourceClass::GetDefaultSourceClass(),0);//hardcoaded prev to FM
SourceID                   clAudioSMEngine::m_enSource_Entertainment_Active   = SourceID(clFactory_AudioSourceClass::GetInvalidSourceClass(),0);
SourceID                   clAudioSMEngine::m_enSource_Entertainment_LSM      = SourceID(clFactory_AudioSourceClass::GetInvalidSourceClass(),0);
SourceID                   clAudioSMEngine::m_enSource_Current_Active         = SourceID(clFactory_AudioSourceClass::GetInvalidSourceClass(),0);
SourceID                   clAudioSMEngine::m_enSource_Previous_Active        = SourceID(clFactory_AudioSourceClass::GetDefaultSourceClass(),0);

time_t                     clAudioSMEngine::m_stInitTime                      = time(NULL);


tVoid clAudioSMEngine::Invalidate()
{
   clAudioSourceFactory::Invalidate();
   clFactory_AudioSourceClass::Invalidate();
   clAudioSourceController::getInstance().vInvalidate();
   m_u8ErrorCode = AUDIOSM_OK;
   m_u8AudioRouteMgr_SrvState = 0;//AMT_C_U8_SVCSTATE_NOT_AVAILABLE;
   m_u8Runlevel                      = 0;
   m_enSource_Entertainment_Previous = SourceID(clFactory_AudioSourceClass::GetInvalidSourceClass(),0);
   m_enSource_Entertainment_Active   = SourceID(clFactory_AudioSourceClass::GetInvalidSourceClass(),0);
   m_enSource_Entertainment_LSM      = SourceID(clFactory_AudioSourceClass::GetInvalidSourceClass(),0);
   m_stInitTime                      = time(NULL);
}


/*!************************************************************************
 * METHOD:        constructor
 *
 * CLASS:         clAudioSMEngine
 *
 * DESCRIPTION:      default constructor.
 *
 * PARAMETER:
 *             none
 *
 * RETURNVALUE:
 *             none
 * HISTORY:
 *  19.09.07       CM-DI/PJ-VW34 Steilmann
 *  Initial Revision
 *************************************************************************/
clAudioSMEngine::clAudioSMEngine()
{


}



/*!***********************************************************************
 * METHOD:        destructor
 *
 * CLASS:         clAudioSMEngine
 *
 * DESCRIPTION:      destructor.
 *
 * PARAMETER:
 *             none
 * RETURNVALUE:
 *             none
 * HISTORY:
 *  19.09.07       CM-DI/PJ-VW34 Steilmann
 *  Initial Revision
 ************************************************************************/
clAudioSMEngine::~clAudioSMEngine()
{}



clAudioSMEngine::enErrorCode clAudioSMEngine::Source_On(SourceID srcID, tU32 u32UserData)
{
  ETG_TRACE_USR1((" Source_On: ignoring SinkID, hardcoded 1, u32UserData= %d",u32UserData));
  return clAudioSMEngine::Source_On(srcID, 1, u32UserData);
}

//new one, no more defaults allowed
clAudioSMEngine::enErrorCode clAudioSMEngine::Source_On(SourceID srcID, tU16 u16SinkID, tU32 u32UserData)
{
   if(((m_u8Runlevel < 2))) //
   {
      ETG_TRACE_USR1((" Source_On:  NOT allowed due to runlevel %d", m_u8Runlevel));
      return AUDIOSM_ERROR;
   }
   if(! clFactory_AudioSourceClass::isValidSourceClass(srcID.enSourceClass))
   {
      ETG_TRACE_ERR(("Source_Off: SourceClass %d not available by AudioSourceFactory",
            srcID.enSourceClass));
      return AUDIOSM_ERROR;
   }
   //return AudioSourceOn(srcID, FALSE, u32UserData);
   return AudioSourceOn(srcID, u16SinkID, FALSE, u32UserData);
}

clAudioSMEngine::enErrorCode clAudioSMEngine::Source_Off(SourceID source, tU32 u32UserData)
{
  ETG_TRACE_USR1((" Source_Off: ignoring SinkID, hardcoded 1, u32UserData= %d",u32UserData));
  return clAudioSMEngine::Source_Off(source, 1, u32UserData);
}

clAudioSMEngine::enErrorCode clAudioSMEngine::Source_Off(SourceID source, tU16 u16SinkID, tU32 u32UserData)
{
   if(((m_u8Runlevel < 2))) //
   {
      ETG_TRACE_USR1((" Source_Off:  NOT allowed due to runlevel %d", m_u8Runlevel));
      return AUDIOSM_ERROR;
   }
   if(! clFactory_AudioSourceClass::isValidSourceClass(source.enSourceClass))
   {
      ETG_TRACE_ERR(("Source_Off: SourceClass %d not available by AudioSourceFactory",
            source.enSourceClass));
      return AUDIOSM_ERROR;
   }
   if(bAudioSourceOff(source, u16SinkID, u32UserData))
   {
    return AUDIOSM_OK;
   }
   else
   {
    return AUDIOSM_ERROR;
   }
}



clAudioSMEngine::enErrorCode clAudioSMEngine::Source_Removed(SourceID source)
{
   // ignore request, as long as runlevel for media not reached
   if(m_u8Runlevel >= 3)
   {
      if(! clFactory_AudioSourceClass::isValidSourceClass(source.enSourceClass))
      {
         ETG_TRACE_ERR(("Source_Removed: SourceClass %d not available by AudioSourceFactory",
               source.enSourceClass));
         return AUDIOSM_ERROR;
      }
      return SourceRemoved(source);
   }
   else
   {
      ETG_TRACE_USR1((" Source_Removed:  Ignored because of runlevel %d", m_u8Runlevel));
   }
   return AUDIOSM_ERROR;
}

clAudioSMEngine::enErrorCode clAudioSMEngine::Source_Insert(SourceID source)
{
   // ignore request, as long as runlevel for media not reached
   if(m_u8Runlevel >= 3)
   {
      if(! clFactory_AudioSourceClass::isValidSourceClass(source.enSourceClass))
      {
         ETG_TRACE_ERR(("Source_Insert: SourceClass %d not available by AudioSourceFactory",
               source.enSourceClass));
         return AUDIOSM_ERROR;
      }
      return SourceInserted(source);
   }
   else
   {
      ETG_TRACE_USR1((" Source_Insert:  Ignored because of runlevel %d", m_u8Runlevel));
   }
   return AUDIOSM_ERROR;
}


tVoid clAudioSMEngine::ResetStack()
{
   clAudioSourceController::getInstance().vResetStack();
}



tVoid clAudioSMEngine::TraceStack()
{
   clAudioSourceController::getInstance().vTraceStack();
}

tVoid clAudioSMEngine::TraceBackupStack()
{
   clAudioSourceController::getInstance().vTraceBackupStack();
}


/*************************************************************************
* METHOD:          vMediaSourceInserted
*
* DESCRIPTION:     A media source has been inserted
*                  So it should be started (regarding priority rules)
*                  This event is currently handled the same way like a
*                  SetSource() request from user.
* PARAMETER:       -u8MediaSource   The inserted media source
*
* RETURNVALUE:     -
*
************************************************************************/
clAudioSMEngine::enErrorCode clAudioSMEngine::SourceInserted(SourceID srcID)
{
   ETG_TRACE_USR4(("SourceInserted: SourceClass: %d subID %#x(%d)"
         , srcID.enSourceClass
         , srcID.u16SubSource
         , srcID.u16SubSource));

   //daw2hi 24.08.2016 get the default sink and use that
   const clSourceClass& srcClass = clFactory_AudioSourceClass::GetSourceClass(srcID.enSourceClass);
   //const sourceClassID invalidSrcClassID = clFactory_AudioSourceClass::GetInvalidSourceClass();

   if(srcClass.getClassID() != clFactory_AudioSourceClass::GetInvalidSourceClass())
   {
      return AudioSourceOn(srcID,srcClass.defaultSink,false,0);
   }
   else
   {
      ETG_TRACE_ERR(("!!! ERROR srcClass not found for Source %d#%d",srcID.enSourceClass,srcID.u16SubSource));
      return AUDIOSM_NOT_ALLOWED;
      //return AudioSourceOn(srcID,1,false,0);
   }

}
/*************************************************************************
* METHOD:          vMediaSourceRemoved
*
* DESCRIPTION:     If a MediaSource is removed/ejected by the user, we have
*                  to stop it if it is currently playing.
*                  Then we start another AudioSource, regarding the rules:
*                  If the previously played Source was Radio: start Radio
*                  If the previously played Source was Media:
*                     If the Media is still available: start that media
*                     If the Media is not available: start Radio
*
* PARAMETER:       tU32 u32MediaSourceToRemove
*                       the media source that was ejected/removed by the user
* RETURNVALUE:     -
*
************************************************************************/
clAudioSMEngine::enErrorCode clAudioSMEngine::SourceRemoved(SourceID srcID)
{

  clAudioSource* pSrc = clAudioSourceFactory::getAudioSource(srcID);
  //Set source to unavailable
  if(pSrc==NULL)
  {
    ETG_TRACE_ERR(("ERROR SourceRemoved: no Instance for SourceClass: %d subID %#x(%d)"
        , srcID.enSourceClass
        , srcID.u16SubSource
        , srcID.u16SubSource));
    return clAudioSMEngine::AUDIOSM_ERROR;
  }
  if(pSrc->enIsSourceAvailable() != clAudioSource::not_available)
  {
    pSrc->vSourceAvailablilityChange(clAudioSource::not_available,clAudioSource::nomedia);
  }

  ETG_TRACE_USR4(("SourceRemoved: SourceClass: %d subID %#x(%d)"
      , srcID.enSourceClass
      , srcID.u16SubSource
      , srcID.u16SubSource));

  /*********************************************************************************************
   * vdu1kor : Remove connection entry from database
   *********************************************************************************************/
  clGeniviAudioSource* pGAS = clGeniviAudioCtrlAdapter::getAudioSource(srcID);

  if( (pGAS != NULL) && (pGAS->u16GetGeniviDomainID() == 1) )
  {
      //special handling if source belongs to domain 1
      std::vector<mainConnectionEntry_s> mainConnectionEntries = clGeniviAudioCtrlAdapter::getMainConnectionEntries();
      am_mainConnectionID_t impactedMainConnectionID=0;
      if(clGeniviAudioCtrlAdapter::getMainConnectionOfSourceSinkFromInternalData(pGAS->u16GetGeniviSourceID(),
              (am_sinkID_t)0, impactedMainConnectionID) && impactedMainConnectionID!=0)
      {
          ETG_TRACE_USR4(("Impacted mainConnectionID %d",impactedMainConnectionID));
      }

      std::vector<am_mainConnectionID_t> listMainConnectionID;
      if(clGeniviAudioCtrlAdapter::bIsSourcOrSinkPartOfMainConnection(pGAS->u16GetGeniviSourceID(),
              (am_sinkID_t)0, listMainConnectionID))
      {
          for(unsigned int i=0;i<listMainConnectionID.size();i++)
          {
              ETG_TRACE_USR4(("Impacted mainConnectionID %d call disconnect",listMainConnectionID[i]));
              clGeniviAudioCtrlAdapter::callHookUserDisconnectionRequest(listMainConnectionID[i]);
          }
      }

	  clGeniviAudioCtrlAdapter::removeMainConnection(pGAS->u16GetGeniviSourceID());
	  ETG_TRACE_USR4(("SourceRemoved: connection entry removed from database"));
  }
  else if((pGAS != NULL) && (pGAS->u16GetGeniviDomainID() != 1))
  {
      auto registeredSinks = clGeniviAudioCtrlAdapter::getRegisteredSinks();
      for (auto it= registeredSinks.begin(); it!= registeredSinks.end(); ++it)
      {
          if((it->second).domainID == 1)
            {
              clGeniviAudioCtrlAdapter::removeMainConnection(pGAS->u16GetGeniviSourceID(), (it->second).sinkID);
            }
      }
    ETG_TRACE_USR4(("removeMainConnection not called for domainID == 101 here, it will be called in CAmControlSenderBase::cbAckDisconnect"));
    ETG_TRACE_USR4(("SourceRemoved: connection entry removed from database"));
  }
  else //pGAS is NULL
  {
    ETG_TRACE_USR4(("SourceRemoved: clGeniviAudioSource is NULL, Failed to removeMainConnection from DB !!!!"));
  }

  /*
   * Delete source if not attached to stack & diagnosis session is not active (PSARCC30-675/PSARCC21-3180)
   */
  if(clAudioSourceController::getInstance().bIsTopOfBackUpStackMuteAndOn())
  {
    ETG_TRACE_USR4(("SourceRemoved: Diagnosis session active"));
    if(!clAudioSourceController::bIsSourceOnStack(srcID))
    {
      ETG_TRACE_USR4(("SourceRemoved: Source not attached to New/Active Stack"));
      if(!clAudioSourceController::bIsSourceOnBackUpStack(srcID)){
	 if((pSrc!=NULL) && ((pSrc->getSourceClass().getRegistrationMode() != clSourceClass::registerExtDynamic)&&(pSrc->getSourceClass().getRegistrationMode() != clSourceClass::registerExtStatic))){
         ETG_TRACE_USR4(("SourceRemoved: Source not attached to New & BackUp Stack, delete it"));
        clAudioSourceFactory::removeAudioSources(srcID);
		}
        return clAudioSMEngine::AUDIOSM_OK;
      }
      else{
        /*
         * During diagnosis session, if source removed which is present on BackUp stack, No need to delete source
         */
        ETG_TRACE_USR4(("SourceRemoved: Source attached to BackUp Stack, No need to delete it"));
        return clAudioSMEngine::AUDIOSM_OK;
      }
    }
    else
    {
      ETG_TRACE_USR4(("SourceRemoved: Source is attached to New/Active Stack"));
    }

  }
  else  // Diagnosis session not active
  {
    /*
     * Only if the source is not attached to Stack we can destruct it
     * it may be added to stack recently and still in Off state because former source
     * is still ramping down
     */

    ETG_TRACE_USR4(("SourceRemoved: Diagnosis session not active"));
    if(!clAudioSourceController::bIsSourceOnStack(srcID))
    {
	  if((pSrc!=NULL) && ((pSrc->getSourceClass().getRegistrationMode() != clSourceClass::registerExtDynamic)&&(pSrc->getSourceClass().getRegistrationMode() != clSourceClass::registerExtStatic))){
      ETG_TRACE_USR4(("SourceRemoved: Source not attached to Stack, delete it"));
      clAudioSourceFactory::removeAudioSources(srcID);
	  }
      return clAudioSMEngine::AUDIOSM_OK;
    }
    //daw2hi 18..6.2018 added for multisink if source removed from other stack than sink 1 (currently no previous/default concept there, go for empty stack)
    else
    {
      if(!clAudioSourceController::bIsSourceOnStack(srcID,1))
      {
        std::vector<int> sinkList;
        if(clAudioSourceController::GetSinkIDsForSource(srcID, sinkList))
        {
          for(unsigned int idx=0;idx<sinkList.size();idx++)
          {
        	  am_Sink_s sinkData;
        	  (void)clGeniviAudioCtrlAdapter::GetGeniviCtrlIF()->getSinkInfoDB(sinkList[idx],sinkData);

        	  if(   (sinkData.name == "AMP_A")
        		 || (sinkData.name == "AMP_B")
				 || (sinkData.domainID == 1) )
        	  {
                  ETG_TRACE_USR4(("bAudioSourceOff for sink %s",sinkData.name.c_str()));
                  bAudioSourceOff(srcID, (tU16)sinkList[idx],(tU32)0);
        	  }
        	  else
        	  {
        		  ETG_TRACE_USR4(("bAudioSourceOff not for sink name %s",sinkData.name.c_str()));
        	  }


          }
        }
        return clAudioSMEngine::AUDIOSM_OK;
      }
    }

  }

  /*
   * vdu1kor - Fix JAC2-4143
   * if active source is removed then source should be changed to m_enSource_Entertainment_Previous
   * otherwise source in stack which is not active is removed hence no need to fallback to previous source
   * as active source is under rampup process
   */
  if(m_enSource_Entertainment_Active !=  srcID){

    if(m_enSource_Entertainment_Active == m_enSource_Current_Active)
    {
      /*
       * Active source must be on stack
       * Ex:
       *   1. Media is on both stack
       *   2. Diagnosis session is ended -> media replaced by FM (m_enSource_Current_Active-FM)
       *   3. On back up stack media source moved to SrcAvailCheck_On
       *   4. FM is not present in current stack
       */
      if(clAudioSourceController::bIsSourceOnStack(m_enSource_Current_Active))
      {
        //Since source removed was not active means active src is still under rampup hence no need to do anything
        ETG_TRACE_USR4(("SourceRemoved: m_enSource_Entertainment_Active, SourceClass: %d subID %d"
            , m_enSource_Entertainment_Active.enSourceClass, m_enSource_Entertainment_Active.u16SubSource));
        ETG_TRACE_USR4(("SourceRemoved: Source removed is not active source, means active src is still  " \
            "under rampup hence no need to do fallback to previous source !!"));
        return AUDIOSM_OK;
      }
    }
    else if((pSrc->getSourceType().name != "entertain"))
    {
      ETG_TRACE_USR4(("SourceRemoved: Source removed is not active Entertainment source perhaps src removed is  " \
          "not entertainment source, no need to do fallback to previous source  !!"));
      /* This handling not required because source component (i.e Phone) will always send
       * SourceRouteRequest(off) if source bacame UnAvailable
       * when source removed is not entertainment source then source should be send to Off state and removed from stack
       vAudioSourceOff(srcID);
       */
      return AUDIOSM_OK;
    }
  }

  return RestorePreviousSource(srcID);
}

clAudioSMEngine::enErrorCode clAudioSMEngine::RestorePreviousSource(SourceID srcID)
{

  ETG_TRACE_USR4(("RestorePreviousSource: entered"));

  SourceID restoreSrcID = m_enSource_Entertainment_Previous;

  clAudioSource* poAudioSource = NULL;
  poAudioSource = clAudioSourceFactory::getAudioSource(restoreSrcID,0,false);
  if(poAudioSource != NULL)
  {
  std::string PreviousSrc(poAudioSource->pacGetName());
  //Check if source is tuner then update the Source as in CABIN
  if((clSourceClass::group_tuner == poAudioSource->getSourceClass().getGroupID())
      && ("entertain" == poAudioSource->getSourceType().name) )
  {
    tU16 SinkID = 17;
    clStack * pStack = clAudioSourceController::pcoGetStack(SinkID);
    if(NULL != pStack)
    { 
       clAudioSource* pSrc = pStack->GetTopSource();
       if(pSrc != NULL)
       {//Check if Entertainment Source is present then only the Both sources gets Exchange else 
        //set to Tuner Source for Bug_1439767
          if((clSourceClass::group_tuner == pSrc->getSourceClass().getGroupID())
               && ("entertain" == pSrc->getSourceType().name) )
          {
             ETG_TRACE_USR1(("vAudioSourceOn: SourceClass: %d subID %d is Tuner Entertainment Source, \
                          Updating property SYP_LAST_TUNER_SOURCE",restoreSrcID.enSourceClass,(tU16)restoreSrcID.u16SubSource));
             std::string topSrc(pSrc->pacGetName());
             if(PreviousSrc != topSrc)
             {
                 restoreSrcID = pSrc->sGetId();
                 ETG_TRACE_USR1(("Exchange Both the Sources"));
             }
          }
          else
          {
              ETG_TRACE_USR3(("Cabin getTopOfGatewaySink is Not Entertainment Sources SourceClass: %d hence " \
                              "restore Tuner as in Cockpit SourceClass: %d ",pSrc->sGetId().enSourceClass,restoreSrcID.enSourceClass));
          }
       }
       else
       {
           ETG_TRACE_USR3(("getTopOfGatewaySink is empty"));
       }
    }
    //clGeniviAudioCtrlAdapter::updateSystemPropertyDB(SYP_LAST_TUNER_SOURCE,srcID.enSourceClass);
   }
  }
  else
  {
      ETG_TRACE_USR1(("clAudioSMEngine: SourceClass: %d subID %d not available by AudioSourceFactory hence restore previous"
        , (srcID.enSourceClass)
        , (tU16)srcID.u16SubSource));
  }
  //No need to create new source object if it's dynamic source vdu1kor SUZUKI-15832
  tBool bCreateNewSrcObj = true;
  const clSourceClass& srcClass = clFactory_AudioSourceClass::GetSourceClass(restoreSrcID.enSourceClass);

  if(srcClass.getRegistrationMode() == clSourceClass::registerDynamic)
  {
    bCreateNewSrcObj = false;
    ETG_TRACE_USR4(("RestorePreviousSource: m_enSource_Entertainment_Previous is of type dynamic"));
  }

  clAudioSource* prevAudioSrc = NULL;
  if(!bIsSourceAvailable(restoreSrcID,bCreateNewSrcObj))
  {
    sourceClassID defClass = clFactory_AudioSourceClass::GetDefaultSourceClass();
    prevAudioSrc = clAudioSourceFactory::getAvailableAudioSource(defClass);
    if(prevAudioSrc == NULL)
      prevAudioSrc = clAudioSourceFactory::getAvailableAudioSource();

    if(prevAudioSrc != NULL)
      restoreSrcID = prevAudioSrc->sGetId();
  }
  else
  {
    prevAudioSrc = clAudioSourceFactory::getAudioSource(restoreSrcID);
  }

  if(prevAudioSrc == NULL)
  {
    ETG_TRACE_FATAL(("RestorePreviousSource: THERE IS NO AVAILABLE AUDIO SOURCE. CANNOT RESTORE ANY AUDIO SOURCE !!"))
      return AUDIOSM_ERROR;
  }

  ETG_TRACE_USR4(("RestorePreviousSource: m_enSource_Current_Active, SourceClass: %d subID %d"
      , m_enSource_Current_Active.enSourceClass
      , m_enSource_Current_Active.u16SubSource));

  /*
   * vdu1kor : If active source is not Entertainment source removed, it's background source exchange,
   * For background source exchange, no need to apply push/pop action rules (Fix NCG3D-1538)
   */
  if( m_enSource_Current_Active !=  srcID )
  {
    ETG_TRACE_USR4(("RestorePreviousSource: Source removed is not active source, means active src is other than  " \
        "entertainment source, Background source Exchange is required !!!!"));

    prevAudioSrc->vSetSourceBG(TRUE);
  }
  else
  {
    ETG_TRACE_USR4(("RestorePreviousSource: Source removed is active source, activating previous source, SourceClass: %d subID %d"
        , restoreSrcID.enSourceClass
        , restoreSrcID.u16SubSource));
  }

  prevAudioSrc->vSetPrevSrcRemoved(TRUE);
  ETG_TRACE_USR4(("RestorePreviousSource: activating m_enSource_Entertainment_Previous, SourceClass: %d subID %d"
      , restoreSrcID.enSourceClass
      , restoreSrcID.u16SubSource));

  //AudioSourceOn(restoreSrcID, TRUE);
  // no more with defaults
  AudioSourceOn(restoreSrcID,1, TRUE, 0);

  if(prevAudioSrc->bIsSourceBG())
    prevAudioSrc->vSetSourceBG(FALSE);

  if(prevAudioSrc->bIsPrevSourceRemoved())
    prevAudioSrc->vSetPrevSrcRemoved(FALSE);

  return AUDIOSM_OK;
}

/*************************************************************************
* METHOD:         vHandleSrcActError
*
* DESCRIPTION:    Remove defect Source from AudioStack.
************************************************************************/
tVoid clAudioSMEngine::vHandleSrcActError(SourceID errSrc)
{

//   SourceID prevAudioSrcID = m_enSource_Entertainment_Previous;
//   if(!bIsSourceAvailable(prevAudioSrcID))
//   {
//      sourceClassID defClass = clFactory_AudioSourceClass::GetDefaultSourceClass();
//      clAudioSource* pAudioSrc = clAudioSourceFactory::getAvailableAudioSource(defClass);
//      if(pAudioSrc == NULL)
//         pAudioSrc = clAudioSourceFactory::getAvailableAudioSource();
//   }

   clAudioSourceController::getInstance().vHandleSrcActError(errSrc);
}

/*************************************************************************
* METHOD:         vAudioSourceOn
*
* DESCRIPTION:    Called on each request of a source change.
*                 This method handles requests for Entertainment-sources
*                 as well as requests for Foreground- and Mix-sources
*
*                 FOREGROUND SOURCES:
*                 - Just forward the request to the AudioSourceController
*                 - No update of Datapool entries
*                 - No update play history
*
*                 MIX SOURCES:
*                 - Just forward the request to the AudioSourceController
*                 - No update of Datapool entries
*                 - No update play history
*
*                 ENTERTANMENT SOURCES:
*                 - Immediately and synchronously update the Datapool in order
*                 to initiate the screen change and to start the visual
*                 animation. This operation has to be performed, independent
*                 of the operation on the AudioStack. I.e. we even update the
*                 Datapool.CurrentSource although we still do not know, if the
*                 new requested source is allowed to play.
*                 This is essential for entertainment sources as it might be
*                 required to change screens and menu-nets for entertainment-
*                 source while other (Foreground-)audiosources are active.
*                 In order to store the playhistory for LSM and PSM, the
*                 Datapool entries:
*                    DPAVDC__CUR_AUDIO_SRC
*                    DPAVDC__PREV_AUDIO_SRC
*                 are stored. Thes entries only contain entertainment sources!
*
* PARAMETER:       -
*
* RETURNVALUE:     -
*
************************************************************************/
// no more with default
clAudioSMEngine::enErrorCode clAudioSMEngine::AudioSourceOn(SourceID srcID, tU16 sinkID, tBool bRemoveCurrent, tU32 u32UserData)
{
  clAudioSource* poAudioSource = NULL;
  //   clSourceClass::group_t enGroup;
  //   clSourceClass::type_t enType;

  // u8AudioSource = static_cast<AudioSources::enAudioSources>(clAudioSourceFactory::mapVirtualToLogicalSource(static_cast<AudioSources::enAudioSources>(u8AudioSource)));
  if(! clFactory_AudioSourceClass::isValidSourceClass(srcID.enSourceClass))
  {
    ETG_TRACE_ERR(("clAudioSMEngine::AudioSourceOn,SourceClass %d not available by AudioSourceFactory",
        srcID.enSourceClass));
    return AUDIOSM_ERROR;
  }

  ETG_TRACE_USR4(("AudioSourceOn: SourceClass: %d subID %d, SourceID %d, UsrData 0x%x"
      , (srcID.enSourceClass)
      , (tU16)srcID.u16SubSource
      , clFactory_AudioSourceClass::GetSourceClass(srcID.enSourceClass).ExtID
      , (tU32)u32UserData));

  /*
   * vdu1kor (fix : NCG3D-5327)
   * If source is dynamic and source not_available is reported earlier then source object is deleted
   * After source become unAvailable, if sourceRouteRequest() is received then it should be ignored
   * For registerDynamic source no need to create object.
   * For registerStatic(Ex AM,FM) & registerNone(all mute sources), object should be created
   */
  const clSourceClass& SrcClass = clFactory_AudioSourceClass::GetSourceClass(srcID.enSourceClass);

  if(SrcClass.getRegistrationMode() == clSourceClass::registerDynamic){
    ETG_TRACE_USR1(("AudioSourceOn: it's Dynamic source, No need to create object if not found"));
    poAudioSource = clAudioSourceFactory::getAudioSource(srcID,0,false);
  }
  else{
    ETG_TRACE_USR1(("AudioSourceOn: it's not Dynamic source"));
    poAudioSource = clAudioSourceFactory::getAudioSource(srcID);
  }

  if(poAudioSource == NULL)
  {
    ETG_TRACE_USR1(("AudioSourceOn: SourceClass: %d subID %d not available by AudioSourceFactory"
        , (srcID.enSourceClass)
        , (tU16)srcID.u16SubSource));
    return AUDIOSM_ERROR;
  }

  clSourceClass oSrcClass = poAudioSource->getSourceClass();
  ETG_TRACE_USR1(("Check Data: SourceClass = %p",&oSrcClass));

  //ToDo: can this be improved

  tBool bSinkAllowed = false;
  std::vector<tU16>::const_iterator it;

#if 0
  int index=0;
  for(it=oSrcClass.Sinks.begin(); it != oSrcClass.Sinks.end(); ++it)
  {
    ETG_TRACE_USR1(("it points to %p and &(oSrcClass.Sinks[%d])=%p",it,index,&(oSrcClass.Sinks[index])));
    index++;
  }
#endif

  for(it=poAudioSource->getSourceClass().Sinks.begin(); it != poAudioSource->getSourceClass().Sinks.end(); ++it)
  {
    if((*it) == sinkID)
    {
      bSinkAllowed = true;
      break;
    }
  }
  if(bSinkAllowed == false)
  {
    ETG_TRACE_USR1(("AudioSourceOn: not allowed for Sink %d",sinkID));
    return AUDIOSM_ERROR;
  }

  poAudioSource->vSetCurActSink(sinkID);
  //   enGroup = poAudioSource->getSourceClass().getGroupID();
  //   enType  = poAudioSource->getSourceClass().getTypeID();

  if (clAudioSourceController::getInstance().enCheckSource(poAudioSource) != clSourceClass::allowed)
  {
    ETG_TRACE_USR1(("SourceClass: %d not allowed!",
        (poAudioSource->sGetId().enSourceClass)));
    return AUDIOSM_ERROR;
  }

  //For available and unknown we will put the source on stack
  if(poAudioSource->enIsSourceAvailable() == clAudioSource::not_available)
  {
    ETG_TRACE_USR1(("ERROR: SourceClass: %d subID %#x(%d): NOT AVAILABLE"
        , (srcID.enSourceClass)
        , (tU16)srcID.u16SubSource
        , (tU16)srcID.u16SubSource));
    return AUDIOSM_ERROR;
  }

  // ******
  // STEP 1
  // ******
  // Update the persistent history-stack of (persistent) Entertainment-Sources
  // Introduced a new configurable parameter "previous". Store previous history if parameter set to "YES".
  // Earlier this was checked by "restorable" parameter.

  /* vdu1kor
   * Only entairtainment source can be cosidered as previous source to be played instead of All sources (fix for JAC2-3489)
   * poAudioSource->getSourceType().previous is always "TRUE" if not defined in source_priorities.xml
   */
  if ((poAudioSource->getSourceType().previous) && (poAudioSource->getSourceType().name == "entertain") && (sinkID == 1) )
  {
    if (srcID != m_enSource_Entertainment_Active)
    {
      if (bRemoveCurrent)
      {
        // if current bg-source should be removed, don't keep it in history-stack
        sourceClassID srcClass = clFactory_AudioSourceClass::GetInvalidSourceClass();
        m_enSource_Entertainment_Previous = SourceID(srcClass,0);
      }
      else
      {
        // the current bg- source will be saved in the history-stack as the previous source
        m_enSource_Entertainment_Previous = m_enSource_Entertainment_Active;
      }
      m_enSource_Entertainment_Active = srcID;

      ETG_TRACE_USR1(("Entertainment Source update: ACTIVE SourceClass: %d subID %d, PREVIOUS SourceClass: %d subID %d"
          , m_enSource_Entertainment_Active.enSourceClass
          , m_enSource_Entertainment_Active.u16SubSource
          , m_enSource_Entertainment_Previous.enSourceClass
          , m_enSource_Entertainment_Previous.u16SubSource));
    }
  }

  //Check if source is tuner entertainment source then update system property SYP_LAST_TUNER_SOURCE (CMG3G-15357)
  if((clSourceClass::group_tuner == poAudioSource->getSourceClass().getGroupID())
      && ("entertain" == poAudioSource->getSourceType().name) )
  {
    ETG_TRACE_USR1(("vAudioSourceOn: SourceClass: %d subID %d is Tuner Entertainment Source, \
                   Updating property SYP_LAST_TUNER_SOURCE",(srcID.enSourceClass),(tU16)srcID.u16SubSource));

    clGeniviAudioCtrlAdapter::updateSystemPropertyDB(SYP_LAST_TUNER_SOURCE,srcID.enSourceClass);
  }

  /* Remember the current active source -> not only entertainment source but any source */

  // daw2hi: but only for sink 1
  if(sinkID == 1) {
    vSetCurrentActiveSource(srcID,(tU16)clGeniviAudioCtrlAdapter::GetSinkIndex(sinkID));
  }


  // ******
  // STEP 2
  // ******
  if(sinkID == 1) {
    // daw2hi: check for MUTE_ENTERTAIN and store persistent Mute
    vDPPersistentMute(poAudioSource, PersMuteDP_Set);
  }
  //daw2hi 18.05.2018 activate this
    //Store persistently
    if((poAudioSource->getSourceType().restorable) && (poAudioSource->bIsSourceRestorable()))
    {
      if(sinkID == 1) 
      	m_enSource_Entertainment_Active = srcID;
      // Update the DP values, which will trigger the screen-changes and animations
      ETG_TRACE_USR3(("vAudioSourceOn(): Step 2: Store Last Source SourceClass: %d subID %d, sink %d"
          , m_enSource_Entertainment_Active.enSourceClass
          , (tU16)m_enSource_Entertainment_Active.u16SubSource, sinkID));

      //Check if Diag Session is active (PSARCCB-8911)
      //if(false == clAudioSourceController::getInstance().bIsTopOfBackUpStackMuteAndOn())
      am_SystemProperty_s systemPropertyDiag = {SYP_DIAG_REMOTE_CONTROL,0};
      if(    (E_OK == clGeniviAudioCtrlAdapter::getSystemPropertyValue(systemPropertyDiag))
          && (systemPropertyDiag.value != 0) )
      {
          ETG_TRACE_USR4(("AudioSourceOn():Diagnosis session is active, No need to update LSM in datapool !"));
      }
      else
      {
        ETG_TRACE_USR4(("AudioSourceOn():Diagnosis session is not active, Store Last Source in datapool for Sink %d!",sinkID));
        vWriteDataPool(srcID.enSourceClass,srcID.u16SubSource, sinkID);

      }

    }
    else {
      // Update the DP values, which will trigger the screen-changes and animations
      ETG_TRACE_USR3(("AudioSourceOn(): Step 2: Ignored due to SourceClassTypeID %d"
          , poAudioSource->getSourceType().id));
    }

  // ******
  // STEP 3
  // ******
  // Request Source Change on AudioSourceController
  if (!(poAudioSource->getSourceType().stackable))
  {
    /*
     * Fix NCG3D-10801
     * if source is not stackable and already on stack, Need to process the source_change(on) request
     */
    if((clFactory_AudioSourceClass::GetSourceSettings().AllowSrcAlreadyonStack)
        && (clAudioSourceController::getInstance().bIsSourceOnStack(srcID,sinkID)))
    {
      /*
       * To handle below use case, when source change request for Media is received
       * AST:  --------TOP--------
       * AST:     TUNER_FM : off
       * AST:   MEDIA_PLAYER#1 : rampdownToOff
       * AST:  ------BOTTOM-------
       */
      if(clAudioSourceController::getInstance().bIsAnotherEntSrcOnStack(srcID))
      {
        if(clAudioSourceController::getInstance().bSELECT(srcID, sinkID, u32UserData))
          return AUDIOSM_OK;
      }

      //source on stack and there is no other entertainment source in stack
      if(clAudioSourceController::getInstance().bON(srcID, sinkID, u32UserData))
        return AUDIOSM_OK;
    }
    /*
     * if source is not stackable and not present on stack, then source will be replaced with one already present
     */
    if(clAudioSourceController::getInstance().bSELECT(srcID, sinkID, u32UserData))
    {
      return AUDIOSM_OK;
    }
  }
  else {
    if (!(poAudioSource->getSourceType().exclusive))
    {
      ETG_TRACE_USR3(("AudioSourceOn(): call bMIX for sinkID %d",sinkID));
      if(clAudioSourceController::getInstance().bMIX(srcID, sinkID, u32UserData))
      {
        return AUDIOSM_OK;
      }
    }
    else
    {
      if(clAudioSourceController::getInstance().bON(srcID, sinkID, u32UserData))
      {
        return AUDIOSM_OK;
      }
    }
  }
  return AUDIOSM_ERROR;
}


clAudioSMEngine::enErrorCode clAudioSMEngine::Source_AvailabilityChange(SourceID srcID,
                                                        clAudioSource::enSourceAvailability availability,
                                                        clAudioSource::enSourceAvailabilityReason reason)
{
   ETG_TRACE_USR4(("Source_Availability: SourceClass: %d subID %#x(%d), Availability: %d, Reason: %d"
            , (srcID.enSourceClass)
            , (tU16)srcID.u16SubSource
            , (tU16)srcID.u16SubSource
            , ETG_CENUM(clAudioSource::enSourceAvailability, availability)
            , ETG_CENUM(clAudioSource::enSourceAvailabilityReason, reason)
            ));
   clAudioSource* pAudSrc = clAudioSourceFactory::isSourceInstantiated(srcID);
   if(availability == clAudioSource::not_available)
   {
      //prevent re-creating instances for sources that had become not available
      pAudSrc = clAudioSourceFactory::isSourceInstantiated(srcID);
   }else{
      //create/get an Instance
	  if(!pAudSrc)
           pAudSrc = clAudioSourceFactory::getAudioSource(srcID);
   }

   if(pAudSrc != NULL)
   {
       //Set the Availability according received
       pAudSrc->vSourceAvailablilityChange(availability, reason);
   }
   return clAudioSMEngine::AUDIOSM_OK;
}


/*************************************************************************
* METHOD:         u8GetNextAvailableEntertainmentSource
*
* DESCRIPTION:    Iterate through the list of entertainment sources,
*                 defined in the specified sequence and find the next available one.
*
* PARAMETER:      u8CurrentAudioSource
*                 pau8ToggleSequence      defines the sequence descriptor to iterate through
*                 u8SizeToggleSequence    defines the size of the sequence descriptor
* RETURNVALUE:    next available ent-source or NONE if no ent-source
*                 available
*
************************************************************************/
SourceID clAudioSMEngine::GetNextAvailableEntertainmentSource(SourceID srcID)
{
   ETG_TRACE_USR4(("u8GetNextAvailableEntertainmentSource: Find next Source after SourceClass: %d subID %#x(%d) "
            , (srcID.enSourceClass)
            , (tU16)srcID.u16SubSource
            , (tU16)srcID.u16SubSource));
   SourceID nextSrcID ( 0 ,0);
   clAudioSource* pSrc = clAudioSourceFactory::getAvailableAudioSource(srcID.enSourceClass);

   if(pSrc != NULL)
   {
      ETG_TRACE_USR4(("u8GetNextAvailableEntertainmentSource: Next Source: %d subID %d "
                  , (pSrc->sGetId().enSourceClass)
                  , (tU16)pSrc->sGetId().u16SubSource));
      return pSrc->sGetId();
   }else{
      //If we cannot find any available
      while(pSrc == NULL)
      {
         sourceClassID nextSrcClass =
               clFactory_AudioSourceClass::GetDefaultSourceClass();
                              //clTogglingSequences::GetNextSourceClass(srcID.enSourceClass, AudioToggleGroups::RMT_CTRL);
         ETG_TRACE_USR4(("u8GetNextAvailableEntertainmentSource: Search default SourceClass Source: %d "
                           , nextSrcClass));
         pSrc = clAudioSourceFactory::getAvailableAudioSource(nextSrcClass);
      }
      ETG_TRACE_USR4(("u8GetNextAvailableEntertainmentSource: Next Source: %d subID %d "
                        , (pSrc->sGetId().enSourceClass)
                        , (tU16)pSrc->sGetId().u16SubSource));
      return pSrc->sGetId();
   }
}

/*************************************************************************
* METHOD:         vOnAudioServiceStateChanged
*
* DESCRIPTION:    ! Under Construction !
*
*
* PARAMETER:      u8ServiceState
*
* RETURNVALUE:    -
*
************************************************************************/

tVoid clAudioSMEngine::vOnAudioServiceStateChanged(tU16 u16Service, tU8 u8ServiceState)
{

   ETG_TRACE_USR4(("vOnAudioServiceStateChanged Service:%d State:%d",
         u16Service,
         u8ServiceState));
// TODO SERVICE AVAILABILITY
   //
//   if((u16Service == CCA_C_U16_SRV_AUDIO_ROUTE_MGR) && (u8ServiceState == AMT_C_U8_SVCSTATE_NOT_AVAILABLE))
//   {
//      ETG_TRACE_USR1(("Set m_u8AudioRouteMgr_SrvState : Not Available"));
//      m_u8AudioRouteMgr_SrvState = AMT_C_U8_SVCSTATE_NOT_AVAILABLE;
//      ETG_TRACE_USR1((" RESET AUDIO STACK!"));
//      vResetStack();
//   }
//
//   if((u16Service == CCA_C_U16_SRV_AUDIO_ROUTE_MGR) && (u8ServiceState == AMT_C_U8_SVCSTATE_AVAILABLE))
//   {
//      ETG_TRACE_USR1(("Set m_u8AudioRouteMgr_SrvState : Available"));
//      m_u8AudioRouteMgr_SrvState = AMT_C_U8_SVCSTATE_AVAILABLE;
//   }
}

/*************************************************************************
* METHOD:         vAudioSourceStarted
*
* DESCRIPTION:    Callback method from AudioSourceController
*                 Distribute the notification to CMs
*
* PARAMETER:      u8AudioSource
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vAudioSourceStarted(tU8 u8AudioSource) { (void)u8AudioSource; }

/*************************************************************************
* METHOD:         vAudioSourceStopped
*
* DESCRIPTION:    Callback method from AudioSourceController
*                 Distribute the notification to CMs
*
* PARAMETER:      u8AudioSource
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vAudioSourceStopped(tU8 u8AudioSource){ (void)u8AudioSource; }

/*************************************************************************
* METHOD:         vAudioSourceInitStarted
*
* DESCRIPTION:    Callback method from AudioSourceController
*                 Distribute the notification to CMs
*
* PARAMETER:      u8AudioSource
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vAudioSourceInitStarted(tU8 u8AudioSource) { (void)u8AudioSource; }

/*************************************************************************
* METHOD:         vAudioSourceStarted
*
* DESCRIPTION:    Callback method from AudioSourceController
*                 Distribute the notification to CMs
*
* PARAMETER:      u8NewEntertainmentSource
*                 bBackground
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vEntertainmentSourceExchanged(tU8 u8NewEntertainmentSource, tBool bBackground) {(void)u8NewEntertainmentSource;(void)bBackground; }

tVoid clAudioSMEngine::vAudioSourceStartMute(SourceID srcID)
{
  (void)srcID;
  ETG_TRACE_USR4(("clAudioSMEngine::vAudioSourceStartMute() called"));
}
tVoid clAudioSMEngine::vAudioSourceStartUnMute(SourceID srcID)
{
  (void)srcID;
  ETG_TRACE_USR4(("clAudioSMEngine::vAudioSourceStartUnMute() called"));
}
/*************************************************************************
* METHOD:         vAudioSourceOff
*
* DESCRIPTION:    Stop an FG Source
* PARAMETER:      tU8 u8AudioSource    the FG source to stop
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vAudioSourceOff(SourceID srcID, tU32 u32UserData)
{
   // we can just use the bAudioSourceOff function and ignore the return value.
   (void)bAudioSourceOff(srcID, u32UserData);
}
/*************************************************************************
* METHOD:         bAudioSourceOff
*
* DESCRIPTION:    Stop an FG Source
* PARAMETER:      tU8 u8AudioSource    the FG source to stop
*
* RETURNVALUE:    bool bSuccess  informs if the SourceOff was succesfull (true) or not (false)
*
************************************************************************/
tBool clAudioSMEngine::bAudioSourceOff(SourceID srcID, tU32 u32UserData)
{
  // use sink 1 if not provided
  return bAudioSourceOff(srcID, 1, u32UserData);
}

tBool clAudioSMEngine::bAudioSourceOff(SourceID srcID, tU16 sinkID, tU32 u32UserData)
{
   tBool bSuccess = true;
   clAudioSource* poAudioSource = NULL;

  ETG_TRACE_USR4(("bAudioSourceOff: SourceClass: %d subID %d sinkID %d"
            , srcID.enSourceClass
      , srcID.u16SubSource
      , sinkID));

   /*
   * vdu1kor (fix : NCG3D-5327)
   * If source is dynamic and source not_available is reported earlier then source object is deleted
   * After source become unAvailable, if sourceRouteRequest() is received then it should be ignored
   * For registerDynamic source no need to create object.
   * For registerStatic(Ex AM,FM) & registerNone(all mute sources), object should be created
   */
   const clSourceClass& srcClass = clFactory_AudioSourceClass::GetSourceClass(srcID.enSourceClass);

   if(srcClass.getRegistrationMode() == clSourceClass::registerDynamic){
     ETG_TRACE_USR1(("bAudioSourceOff: it's Dynamic source, No need to create object if not found"));
    poAudioSource = clAudioSourceFactory::getAudioSource(srcID,0,false);
   }
   else{
     ETG_TRACE_USR1(("bAudioSourceOff: it's not Dynamic source"));
     poAudioSource = clAudioSourceFactory::getAudioSource(srcID);
   }

   if(poAudioSource == NULL)
   {
      ETG_TRACE_USR1(("bAudioSourceOff: %d subID %d not available by AudioSourceFactory"
            , (srcID.enSourceClass)
            , (tU16)srcID.u16SubSource));
      return false;
   }

   const stSourceType& srcType = poAudioSource->getSourceType();

   poAudioSource->vSetCurActSink(sinkID);
   // Delegate Request to AudioSourceController
   if(!srcType.exclusive)
   {
      clAudioSourceController::getInstance().bMIXOFF(srcID, sinkID, u32UserData);
   }
   else if(srcType.stackable)
   {
    //daw2hi: Remove also PersistentMute
    vDPPersistentMute(poAudioSource, PersMuteDP_Reset);
      clAudioSourceController::getInstance().bOFF(srcID, sinkID, 0);  //daw2hi: 27.11.2018 Bug fixed
   }
   else
   {

      //ToDo: daw2hi 02.08.2016 check if below part shoudl be done only for sink 1

      bSuccess = false;

      if(sinkID == 1)
    {

#if defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_RN_AIVI)
      // CMG3G-10573 Command for HMI to request a source change to previous audio source

      if(bIsSourceAvailable(m_enSource_Entertainment_Previous,false))
      {
         if(AUDIOSM_OK == AudioSourceOn(m_enSource_Entertainment_Previous,1, FALSE, 0))
         {
            bSuccess = true;
         }

      }
      else
      {
         ETG_TRACE_ERR(("previous Source seems no more available, try to get default source"));
         sourceClassID defClass = clFactory_AudioSourceClass::GetDefaultSourceClass(); //default has currently no subsource ??
         clAudioSource* defAudioSrc = clAudioSourceFactory::getAvailableAudioSource(defClass);
         if(defAudioSrc != NULL)
         {
            SourceID  defSrcID = defAudioSrc->sGetId();
            if(AUDIOSM_OK == AudioSourceOn(defSrcID,1, FALSE, 0))
            {
               bSuccess = true;
            }

         }
         else
         {
            ETG_TRACE_ERR(("default source also not available -> rejecting disconnect"));
         }

      }
#else
      ETG_TRACE_ERR(("bAudioSourceOff: ONLY ALLOWED FOR Mix AND Fg SOURCES: conflict SourceClass: %d subID %d"
         , srcID.enSourceClass
         , srcID.u16SubSource));


#endif
      }
      else
    {
      bSuccess = false;
      //#### allow disconnect of Entertain for other sink (e.g. RSE) ####
      bSuccess = clAudioSourceController::getInstance().bOFF(srcID, sinkID, u32UserData);
      //#### allow disconnect end ####
    }
   }
   return bSuccess;
}
/*************************************************************************
* METHOD:         vDiagnosisAudioSourceOn
*
* DESCRIPTION:    Activate an AudioSource for Diagnosis/RemoteControl
*                 AudioSource is started with force-option, i.e. a play
*                 request is sent to FC-Audio without any prio check and
*                 without any media-available check!
*                 The normal AudioStack is frozen.
* PARAMETER:      tU8 u8AudioSource    the Diagnosis Audio source to start
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vDiagnosisAudioSourceOn(tU8 u8AudioSource, tBool bForce)
{
   (void) u8AudioSource;
   (void) bForce;
   ETG_TRACE_ERR(("vDiagnosisAudioSourceOn: actually not supported"));
}

/*************************************************************************
* METHOD:         vDiagnosisAudioSourceOff
*
* DESCRIPTION:    Stop an AudioSource for Diagnosis/RemoteControl
*                 AudioSource is stopped with force-option, i.e. a stop
*                 request is sent to FC-Audio without any prio check!
* PARAMETER:      tU8 u8AudioSource    the Diagnosis Audio source to stop
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vDiagnosisAudioSourceOff(tU8 u8AudioSource, tBool bForce)
{
  (void) u8AudioSource;
  (void) bForce;
   ETG_TRACE_ERR(("vDiagnosisAudioSourceOff: actually not supported"));
}

/*************************************************************************
* METHOD:         vDiagnosisOff
*
* DESCRIPTION:    Stop Diagnosis Mode
*                 If a Diagnosis AudioSource is currently active, it is
*                 stopped.
*                 The normal Audio Stack is recovered.
* PARAMETER:
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vDiagnosisOff()
{
   ETG_TRACE_ERR(("vDiagnosisOff: actually not supported"));
}

/*************************************************************************
* METHOD:         bIsSystemFree
*
* DESCRIPTION:    Asks CM-System, if PIN-Code Lock is active
* PARAMETER:
*
* RETURNVALUE:    tBool: True, if PIN-Code is unlocked, False otherwise
*
************************************************************************/
tBool clAudioSMEngine::bIsSystemFree()
{
   ETG_TRACE_USR1(("bIsSystemFree: TRUE"));
   return TRUE;
}

/*************************************************************************
* METHOD:         bIsEntertainmentSourceAvailable
*
* DESCRIPTION:    there are 2 types of Entertainment sources: Radio and Media
*                 Radio sources are available always.
*                 Media sources availability is given by bIsMediaSourceAvailable()
*
* PARAMETER:       u8AudioSource
*
* RETURNVALUE:     true, if the specified Entertainment source is available
*
*
************************************************************************/
tBool clAudioSMEngine::bIsSourceAvailable(SourceID audioSource,tBool bCreateNewSource)
{
   tBool retVal = FALSE;
   clAudioSource* pAudSrc = clAudioSourceFactory::getAudioSource(audioSource,bCreateNewSource);
   if(pAudSrc != NULL)
   {
      if((pAudSrc->enIsSourceAvailable() == clAudioSource::available)
            ||(pAudSrc->enIsSourceAvailable() == clAudioSource::unkonwn))
      {
         retVal = TRUE;
      }
      ETG_TRACE_USR2(("bIsSourceAvailable: SourceClass: %d subID %d, Availability: %d"
                  , audioSource.enSourceClass
                  , audioSource.u16SubSource
                  , ETG_CENUM(clAudioSource::enSourceAvailability, pAudSrc->enIsSourceAvailable())));
   }else{
      ETG_TRACE_USR1(("bIsSourceAvailable: SourceClass: %d subID %d not available "
                  , audioSource.enSourceClass
                  , audioSource.u16SubSource));
   }

   return retVal;
}

/*************************************************************************
* METHOD:         vSetRunlevel
*
* DESCRIPTION:    This is the entrypoint for CM-Startup, which controls
*                 the powerstates and runlevels of the AudioSystem via
*                 this method.
*                 --------------------------------------------------------
*                 Runlevel 0:
*                    - Audio is muted.
*                    - AudioStack is reseted
*                    - The virtual Source: System-Mute is on stack. This
*                    ensures the audio system to remain muted, even if
*                    source-switch requests are applied.
*                    - AudioSource-ON() is allowed. This is required to enable
*                    early Source-Switch requests. The requested sources
*                    are put to the stack but not yet started
*                 --------------------------------------------------------
*                 Runlevel 1:
*                    - This is the runlevel for limited Phone mode
*                    - All Audiosources except phone are muted
*                 --------------------------------------------------------
*                 Runlevel 2:
*                    - This is the runlevel for limited Entertainment audio
*                    - All Radio Sources are enabled
*                    - All Media Sources are still disabled
*                 --------------------------------------------------------
*                 Runlevel 3:
*                    - This is the runlevel for full audio
*                    - All Audio Sources are enabled
*
*
* PARAMETER:      tU8 u8NewRunlevel
*
* RETURNVALUE:    -
*
************************************************************************/
clAudioSMEngine::enErrorCode clAudioSMEngine::SetRunlevel(tU8 u8NewRunlevel, tBool bColdStart)
{
   clAudioSMEngine::enErrorCode retVal = clAudioSMEngine::AUDIOSM_OK;
   tU8 u8OldRunlevel = m_u8Runlevel;
   //register static AudioSources at first
   if (u8OldRunlevel == 0 && u8NewRunlevel > 0)
   {
      // lets try this first (need this now more earlier)
      (void)clAudioSourceController::getInstance();

      if(!clAudioSourceFactory::Init())
      {
         ETG_TRACE_ERR(("SetRunlevel: ERROR Initializing AudioSource Factory"));
         retVal = clAudioSMEngine::AUDIOSM_ERROR;
         return retVal;
      }
      //on cold start ND load rule set , update sound settings for ll sinks 
      clGeniviAudioCtrlAdapter::vLoadSettings();
   }

   m_u8Runlevel = u8NewRunlevel;

   tU8 au8Data[2]={u8OldRunlevel, u8NewRunlevel};
   ETG_TRACE_USR2(("vSetRunlevel: Old Runlevel: %d New Runlevel: %d"
         ,au8Data[0]
         ,au8Data[1]));

   switch (u8NewRunlevel)
   {
   case 0:
      {
         if (u8OldRunlevel > 0)
         {
           //reset LSM
           for(auto it=clGeniviAudioCtrlAdapter::m_sinkList.begin();it!=clGeniviAudioCtrlAdapter::m_sinkList.end();it++)
           {
               // Reset LSM to false. We might come back without shutdown and have to restore LSM
               // Inf4CV fix no cockpit audio after short Ignition off/on
               ETG_TRACE_USR4(("SetLSM back to false for m_sinkList[%d] : sinkID %d",it->first,it->second.s_sink.sinkID));
               clGeniviAudioCtrlAdapter::SetLSM(it->second.s_sink.sinkID,false);
           }

           // Reset Stack
           vResetStack();
//         AudioSourceOn(SourceID(AudioSources::MUTE_SYSTEM,0));
//         AudioSourceOn(SourceID(AudioSources::MUTE_LIMITED_SYSTEM,0));
         }
      }
      break;
   case 1:
      {
         if (u8OldRunlevel < 1)
         {
            // enable limited phone audio
//            vAudioSourceOff(SourceID(AudioSources::MUTE_PIN, 0));
//            vAudioSourceOff(SourceID(AudioSources::MUTE_SYSTEM,0));
         }
         else if (u8OldRunlevel > 1)
         {
            // No Entertainment in runlevel 1. Only Phone.
            //vResetEntertainmentsource();
//            AudioSourceOn(SourceID(AudioSources::MUTE_LIMITED_SYSTEM,0));
         }
      }
      break;
   case 2:
      {
         if (u8OldRunlevel < 1)
         {
//            vAudioSourceOff(SourceID(AudioSources::MUTE_SYSTEM,0));
         }
         if (u8OldRunlevel < 2)
         {
            // Restore Entertainmentsource (limited to radio) and persistent Pause-state
#if defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_INF4CV) || defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_SMART)
            retVal = vRestoreEntertainmentsource();
#endif 
            vRestorePersistentMuteState();
//            vAudioSourceOff(SourceID(AudioSources::MUTE_LIMITED_SYSTEM,0));
         }
      }
      break;
   case 3:
      {
         if (u8OldRunlevel < 3)
         {
            time(&m_stInitTime);

            clGeniviAudioCtrlAdapter::vSetAmplifierMute(MUTE_NO_AUD_DEV_ON);
            clGeniviAudioCtrlAdapter::vSetMuteOnStartUp(SYP_HMISTARTUP_MUTE_ON,bColdStart);

            // Restore Entertainmentsource (radio and media)and persistent Pause-state
#if defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_INF4CV) || defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_SMART) || defined(VARIANT_S_FTR_ENABLE_UNITTEST)
            retVal = vRestoreEntertainmentsource();
#endif 
            vRestorePersistentMuteState();
            const clSourceClass* srcClass = clFactory_AudioSourceClass::GetSourceClass("MUTE_ZERO_VOLUME");
            if(srcClass != NULL)
            {
              sourceClassID muteClassID = srcClass->getClassID();
              vAudioSourceOff(SourceID(muteClassID,0));
            }
            else
            {
              //Error, no MUTE_ZERO_VOLUME
              ETG_TRACE_ERR(("AudioSourceClass MUTE_ZERO_VOLUME not found !!!"));
            }
//          vAudioSourceOff(SourceID(AudioSources::MUTE_ZERO_VOLUME,0));
//          vAudioSourceOff(SourceID(AudioSources::MUTE_LIMITED_SYSTEM,0));

      //Get AmpAvailability & remove MUTE_NO_AUD_DEV_ON if Present
      if(true == clGeniviAudioCtrlAdapter::bAudioDeviceAvailability())
      {
        ETG_TRACE_USR3(("SetRunlevel, AudioDeviceAvailability is TRUE, Removing MUTE_NO_AUD_DEV"));
        const clSourceClass* srcClass = clFactory_AudioSourceClass::GetSourceClass("MUTE_NO_AUD_DEV");
        if(srcClass != NULL)
        {
          sourceClassID muteClassID = srcClass->getClassID();
            //vAudioSourceOff(SourceID(muteClassID,0),1,u32UserData); // Sink 1

            (void)bAudioSourceOff(SourceID(muteClassID,0), 1, 0); // Sink 1
#ifdef VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_INF4CV
            (void)bAudioSourceOff(SourceID(muteClassID,0), 2, 0); // Sink 2
#endif
        }
      }
         }
      }
      break;
   default:
      // invalid runlevel
      break;
   }
   return retVal;
}

tU8 clAudioSMEngine::GetRunlevel()
{
   return m_u8Runlevel;
}

tVoid clAudioSMEngine::ListSources()
{
   clAudioSourceFactory::ListSources();
}


tVoid clAudioSMEngine::vResetStack()
{
   ETG_TRACE_ERR(("vResetStack called without sink -> using sink 1"));
   clAudioSMEngine::vResetStack(1);
}

tVoid clAudioSMEngine::vResetStack( int sinkID)
{

   clAudioSourceController::getInstance().vResetStack(sinkID);
   // place additionally to a clean stack a Pin-Mute to ensure Pin-lock before
   // XML-Start, because XML locks the PIN-lock too late at startup.
   //TODO check this needed
   //clAudioSourceController::getInstance().bON(SourceID(AudioSources::MUTE_PIN,0), 0);
}

bool clAudioSMEngine::bIsPersistentMuteState()
{
   ETG_TRACE_USR3(("entered bIsPersistentMuteState"));
#if defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_RN_AIVI)
   dp_tclAudioStackDPAudioStack_PersistentMute   oPersMute;
   tU8 u8PersMute = 0;
   tS32 ret = oPersMute.s32GetData(u8PersMute);
   ETG_TRACE_USR3(("bIsPersistentMuteState is %d (ret=%x)",(tU16)u8PersMute,ret));
   if(u8PersMute != 0)
   {
     return true;
   }
   return false;
#else
   return false;
#endif
}

/*************************************************************************
* METHOD:         vRestorePersistentMuteState
*
* DESCRIPTION:    Recovers the Persistent MuteState
*                 The Persistent MuteState is the user-requested Mute
*                 which pauses Entertainmentsources. (MFL, HK-Phone, ...)
* PARAMETER:      -
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vRestorePersistentMuteState()
{
   ETG_TRACE_USR3(("vRestorePersistentMuteState not for all variants used"));
#if defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_PSA) || defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_RN_AIVI)
   dp_tclAudioStackDPAudioStack_PersistentMute   oPersMute;
   tU8 u8PersMute = 0;
   tS32 ret = oPersMute.s32GetData(u8PersMute);
   ETG_TRACE_USR3(("vRestorePersistentMuteState is %d (ret=%x)",(tU16)u8PersMute,ret));
   if(u8PersMute != 0)
   {
     const clSourceClass* pSrcClass_EntMute = clFactory_AudioSourceClass::GetSourceClass("MUTE_ENTERTAIN");
     if(pSrcClass_EntMute)
     {
       ETG_TRACE_USR3(("vRestorePersistentMuteState: Set MUTE_ENTERTAIN"));
     //AudioSourceOn(SourceID(pSrcClass_EntMute->getClassID(),0));
     // daw2hi: no more with defaults (RSE)
     AudioSourceOn(SourceID(pSrcClass_EntMute->getClassID(),0),1,false, 0);
       //need to inform HMI about the Mute
       clGeniviAudioCtrlAdapter::vSetPersistentMute();
     }
   }
#endif
}

time_t  clAudioSMEngine::GetTime_Init()
{
   return m_stInitTime;
}

tVoid clAudioSMEngine::vResetTime_Init()
{
  // Reset the reference Time
  time(&m_stInitTime);
  ETG_TRACE_USR1(("Reset Time_Init to %s",ctime(&m_stInitTime)));
}

/*
 * Function to check if user specific data pool should be loaded for Personalization feature
 * Currently Personalization feature is required only for PIVI
 * Checking OEM type is 'Infiniti' then it's PIVI specific target
 */
bool clAudioSMEngine::bLoadUsrProfileDataPool()
{
//#ifndef LinuxX86Make
//  FILE* dbg_write_ptr = fopen(LOGFILE_CTRL,"a");
//  fprintf (dbg_write_ptr, "Controller: clAudioSMEngine::bLoadUsrProfileDataPool\n");
//  fclose(dbg_write_ptr);
//#endif

    ETG_TRACE_USR1(("Entered bLoadUsrProfileDataPool"));
    tU8 u8VariantInfo = 0;

    if((DP_S32_NO_ERR == DP_s32GetConfigItem("CMVariantCoding", "OEMType", &u8VariantInfo,1)))
    {
//#ifndef LinuxX86Make
//      dbg_write_ptr = fopen(LOGFILE_CTRL,"a");
//      fprintf (dbg_write_ptr, "Controller: clAudioSMEngine::bLoadUsrProfileDataPool\n");
//        fclose(dbg_write_ptr);
//#endif
        if(OEM_TYPE_INFINITI == u8VariantInfo)
        {
//#ifndef LinuxX86Make
//          dbg_write_ptr = fopen(LOGFILE_CTRL,"a");
//          fprintf (dbg_write_ptr, "Controller: clAudioSMEngine::bLoadUsrProfileDataPool P-IVI\n");
//            fclose(dbg_write_ptr);
//#endif
            ETG_TRACE_USR1(("bLoadUsrProfileDataPool: Device is P-IVI"));
            return true;
        }
        else
        {
//#ifndef LinuxX86Make
//          dbg_write_ptr = fopen(LOGFILE_CTRL,"a");
//          fprintf (dbg_write_ptr, "Controller: clAudioSMEngine::bLoadUsrProfileDataPool A-IVI\n");
//            fclose(dbg_write_ptr);
//#endif
            ETG_TRACE_USR1(("bLoadUsrProfileDataPool: Device is A-IVI"));
            return false;
        }
    }
    else
    {
        ETG_TRACE_ERR(("bLoadUsrProfileDataPool: Reading OEMType failed"));
    }
    return false;
}
/*************************************************************************
* METHOD:         vRestoreEntertainmentsource
*
* DESCRIPTION:    Recovers the LSM Entertainmentsource
*                 If runlevel is < 3, a Media-LSM will not be started
* PARAMETER:      -
*
* RETURNVALUE:    -
*
************************************************************************/
#if 1 //my
clAudioSMEngine::enErrorCode clAudioSMEngine::vRestoreEntertainmentsource()
{
  ETG_TRACE_USR4(("vRestoreEntertainmentsource:" ));
  time_t TheTimeIsNow;
  time(&TheTimeIsNow);
  tDouble SecSinceRunlvl3 = difftime(TheTimeIsNow,m_stInitTime);
  ETG_TRACE_USR4(("vRestoreEntertainmentsource: Seconds since Runlevel 3: %f",(etg_tF32)SecSinceRunlvl3 ));

#ifndef VARIANT_S_FTR_ENABLE_AUDIO_STACK_UTEST
  DP_vCreateDatapool();
#endif

  if(clGeniviAudioCtrlAdapter::m_sinkList.size()<=0) return AUDIOSM_ERROR;

	ETG_TRACE_USR4(("vRestoreEntertainmentsource: m_sinkList.size = %d",clGeniviAudioCtrlAdapter::m_sinkList.size()));
  std::map<unsigned int, struct sinkData>::iterator it = clGeniviAudioCtrlAdapter::m_sinkList.begin();
  for(unsigned int i = 0; i<=clGeniviAudioCtrlAdapter::m_sinkIndex && it!= clGeniviAudioCtrlAdapter::m_sinkList.end();it++,i++)
  {
	if((it->second).bRestore == false)
		continue;
		
    tU8 lastSourceClass = 0;
    tU16 u16SubID = 0;
    std::string SrcName;
    SourceID lastSource (lastSourceClass,u16SubID);

    ETG_TRACE_USR4(("vRestoreEntertainmentsource: for sinkID %d, with domain %d and sinkName %s",
        (it->second).s_sink.sinkID, (it->second).s_sink.domainID, (it->second).s_sink.name.c_str()));

    if((it->second).s_sink.sinkID != 0 && clGeniviAudioCtrlAdapter::GetLSM((it->second).s_sink.sinkID)==false)
    {
      //tU16 validSink=0;
      tU16 sinkID = (it->second).s_sink.sinkID;

      vReadDataPool(lastSourceClass,u16SubID,sinkID,SrcName);
      clAudioSource* pAudSrc = NULL;
      ETG_TRACE_USR4(("vRestoreEntertainmentsource: Read from DP SourceClass: %d subID %d for sinkID %d", lastSourceClass, u16SubID, sinkID));
      SourceID lastSource (lastSourceClass,u16SubID);
      pAudSrc = NULL;
      m_enSource_Entertainment_LSM = lastSource;
      if(clAudioSourceFactory::isSourceInstantiated(lastSource) == NULL)
      {
        pAudSrc = clAudioSourceFactory::getAudioSource(lastSource);
        if(pAudSrc != NULL)
        {
          pAudSrc->vSetSourceLSM(true);
          pAudSrc->vSetCurActSink((it->second).s_sink.sinkID);
          pAudSrc->vSourceAvailablilityChange(clAudioSource::unkonwn, clAudioSource::samemedia);
        }
      }
      else
      {
        pAudSrc = clAudioSourceFactory::getAudioSource(lastSource);
        if((pAudSrc!=NULL) && (pAudSrc->getSourceClass().getRegistrationMode() == clSourceClass::registerDynamic))
        {
          if(!pAudSrc->bIsSourceLSM())
          {
            pAudSrc->vSetSourceLSM(true);
            pAudSrc->vSourceAvailablilityChange(clAudioSource::unkonwn, clAudioSource::samemedia);
          }
          pAudSrc->vSetCurActSink((it->second).s_sink.sinkID);
          //validSink = pAudSrc->vSetValidSink((it->second).s_sink.sinkID);
          (void)pAudSrc->vSetValidSink((it->second).s_sink.sinkID);
        }
      }

      if((pAudSrc!=NULL) && (pAudSrc->getSourceType().name != "entertain"))
      {
        ETG_TRACE_USR1(("vRestoreEntertainmentsource: Restored source Not Entertainment Source, Restoring to default" ));
        lastSource.enSourceClass = clFactory_AudioSourceClass::GetDefaultSourceClass();
        lastSource.u16SubSource = 0;
        lastSourceClass = lastSource.enSourceClass;
      }
	  else if(pAudSrc!=NULL)
	  {
			ETG_TRACE_USR1(("vRestoreEntertainmentsource: Restored source %s",pAudSrc->pacGetName()));
	  }
      else
      {
        ETG_TRACE_USR1(("vRestoreEntertainmentsource: Restored source Entertainment Source" ));
      }


      if(lastSourceClass == 0)
      {
        ETG_TRACE_USR1(("vRestoreEntertainmentsource: Restore Default source due to default value of datapool" ));
        lastSource.enSourceClass = clFactory_AudioSourceClass::GetDefaultSourceClass();
        lastSource.u16SubSource = 0;
        SrcName = clFactory_AudioSourceClass::GetDefaultSourceName();
      }

      if (!bIsSourceAvailable(lastSource))
      {
        ETG_TRACE_USR1(("vRestoreEntertainmentsource: Restore Default source due to availability" ));
        lastSource = GetNextAvailableEntertainmentSource(
            SourceID(clFactory_AudioSourceClass::GetDefaultSourceClass(),0));
        //      lastSource.enSourceClass = AudioSources::TUNER_FM;
        //      lastSource.u16SubSource = 0;
          SrcName = clFactory_AudioSourceClass::GetDefaultSourceName();
      }

      ETG_TRACE_USR1(("vRestoreEntertainmentsource: Adding MainConnection SourceClass: %d subID %d"
          , lastSource.enSourceClass
          , lastSource.u16SubSource));
      clAudioSource* pGSrc =  clAudioSourceFactory::getAudioSource(lastSource);
    
      if(pGSrc == NULL )
      {
        ETG_TRACE_ERR(("vRestoreEntertainmentsource: No AudioSource Object found"));
        //return AUDIOSM_ERROR;
      }
      else
      {
#if defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_INF4CV) || defined(VARIANT_S_FTR_ENABLE_UNITTEST)
//da2hi: what about smart here??
        if(E_OK == clGeniviAudioCtrlAdapter::updateConnect(SrcName,sinkID)) // review :get the genivi srcid
          clGeniviAudioCtrlAdapter::SetLSM(sinkID,true);
#else
	AudioSourceOn(lastSource,1,false, 0);
#endif
      }
    }
  }

  return AUDIOSM_OK;
}
#endif

#if 0 // indian
clAudioSMEngine::enErrorCode clAudioSMEngine::vRestoreEntertainmentsource()
{
   ETG_TRACE_USR4(("vRestoreEntertainmentsource:" ));
   time_t TheTimeIsNow;
   time(&TheTimeIsNow);
   tDouble SecSinceRunlvl3 = difftime(TheTimeIsNow,m_stInitTime);
   ETG_TRACE_USR4(("vRestoreEntertainmentsource: Seconds since Runlevel 3: %f",(etg_tF32)SecSinceRunlvl3 ));

#ifndef VARIANT_S_FTR_ENABLE_AUDIO_STACK_UTEST
  DP_vCreateDatapool();
#endif

   if(clGeniviAudioCtrlAdapter::m_sinkList.size()<=0) return AUDIOSM_ERROR;

    ETG_TRACE_USR4(("vRestoreEntertainmentsource:%d",clGeniviAudioCtrlAdapter::m_sinkList.size()));
    // iterate over all the restore able sinks
    std::map<unsigned int, struct sinkData>::iterator it = clGeniviAudioCtrlAdapter::m_sinkList.begin();
    for(int i = 0; i<=clGeniviAudioCtrlAdapter::m_sinkIndex && it!= clGeniviAudioCtrlAdapter::m_sinkList.end();it++,i++)
    {
      tU8 lastSourceClass = 0;
      tU16 u16SubID = 0;
      std::string SrcName;
      SourceID lastSource (lastSourceClass,u16SubID);
      ETG_TRACE_USR4(("vRestoreEntertainmentsource:" ));

      if((it->second).s_sink.sinkID != 0 && clGeniviAudioCtrlAdapter::GetLSM((it->second).s_sink.sinkID)==false)
      {
        tU16 validSink=0, sinkID = (it->second).s_sink.sinkID;
        vReadDataPool(lastSourceClass,u16SubID,sinkID,SrcName);
        clAudioSource* pAudSrc = NULL;
        ETG_TRACE_USR4(("vRestoreEntertainmentsource: Read from DP SourceClass: %d subID %d", lastSourceClass, u16SubID));
        SourceID lastSource (lastSourceClass,u16SubID);
        pAudSrc = clAudioSourceFactory::isSourceInstantiated(lastSource);
        m_enSource_Entertainment_LSM = lastSource;
        // if source is not instantiated that is during startup and during low voltage , source is already instantiated
        //(pAudSrc->getSourceClass().getRegistrationMode() == clSourceClass::registerDynamic) during low voltage
        if(pAudSrc == NULL)
         {
            pAudSrc = clAudioSourceFactory::getAudioSource(lastSource);

            if(pAudSrc!=NULL && pAudSrc->bIsSourceLSM()==false)
            {
                  pAudSrc->vSetSourceLSM(true);
                  pAudSrc->vSourceAvailablilityChange(clAudioSource::unkonwn, clAudioSource::samemedia);
                  pAudSrc->vSetCurActSink((it->second).s_sink.sinkID);
                  validSink = pAudSrc->vSetValidSink((it->second).s_sink.sinkID);
            }
          }

     if((pAudSrc!=NULL && (pAudSrc->getSourceType().name != "entertain"))|| (lastSourceClass == 0))
     {
     ETG_TRACE_USR1(("vRestoreEntertainmentsource: Restored source Not Entertainment Source, Restoring to default" ));
     lastSource.enSourceClass = clFactory_AudioSourceClass::GetDefaultSourceClass();
     SrcName = clFactory_AudioSourceClass::GetDefaultSourceName();
     lastSource.u16SubSource = 0;
     lastSourceClass = lastSource.enSourceClass;
     }
     else
     {
     ETG_TRACE_USR1(("vRestoreEntertainmentsource: Restored source Entertainment Source" ));
     }
       // let update connect decide the source is available or no and then forward to connect
     if(bIsSourceAvailable(lastSource)){
       if(E_OK == clGeniviAudioCtrlAdapter::updateConnect(SrcName,sinkID)) // review :get the genivi srcid
            clGeniviAudioCtrlAdapter::SetLSM(sinkID,true);
     }
       //if mute_no_aud dev is not removed , do not restore for other sinks
      }
      if(false == clGeniviAudioCtrlAdapter::bAudioDeviceAvailability() || (m_u8Runlevel < 3))
           break;
    }

   return AUDIOSM_OK;
}
#endif

void clAudioSMEngine::vStartFM_onSink(tU16 sinkID)
{
  ETG_TRACE_ERR(("HACK HACK"));
  ETG_TRACE_ERR(("clAudioSMEngine::vStartFM_onSink Start FM on Sink %d",sinkID));

  const clSourceClass* pSrcClassFM = clFactory_AudioSourceClass::GetSourceClass("TUNER_FM");
    if(pSrcClassFM==NULL)
    {
      ETG_TRACE_ERR(("Source TUNER_FM not found, abort"));
      return;
    }
  SourceID sourceFM(pSrcClassFM->SourceClassID,(tU16)0);

  clAudioSource* pAudSrc = clAudioSourceFactory::getAudioSource(sourceFM);
  if(pAudSrc!=NULL)
  {
    pAudSrc->addSink(sinkID);
    AudioSourceOn(sourceFM,sinkID,false, 0);

    ETG_TRACE_ERR(("HACK HACK END"));
  }
  else
  {
    ETG_TRACE_ERR(("HACK HACK END could not start FM on Sink %d",sinkID));
  }
}

/************************************************************************
 * FUNCTION     : u8GetSrcGroupID_Previous
 * DESCRIPTION  : function to get previous source group id
 * PARAMETER    : NIL
 * RETURNVALUE  : clSourceClass::sourceGroupID
 * HISTORY      :
 *              04.01.16 Vyankatesh VD  Initial Revision.
 ************************************************************************/
clSourceClass::sourceGroupID clAudioSMEngine::u8GetSrcGroupID_Previous()
{
   return (clFactory_AudioSourceClass::GetSourceClass(m_enSource_Previous_Active.enSourceClass).getGroupID());
}

/************************************************************************
 * FUNCTION     : u8GetSrcGroupID_Current
 * DESCRIPTION  : function to get current source group id
 * PARAMETER    : NIL
 * RETURNVALUE  : clSourceClass::sourceGroupID
 * HISTORY      :
 *              04.01.16 Vyankatesh VD  Initial Revision.
 ************************************************************************/
clSourceClass::sourceGroupID clAudioSMEngine::u8GetSrcGroupID_Current()
{
  return (clFactory_AudioSourceClass::GetSourceClass(m_enSource_Current_Active.enSourceClass).getGroupID());
}

/************************************************************************
 * FUNCTION     : vSetCurrentActiveSource
 * DESCRIPTION  : function to set current active source and previous active source
 * PARAMETER    : AudioSource::SourceID
 * RETURNVALUE  : tVoid
 * HISTORY      :
 *              04.01.16 Vyankatesh VD  Initial Revision.
 ************************************************************************/
tVoid clAudioSMEngine::vSetCurrentActiveSource(AudioSource::SourceID srcID,tU16 sinkIndex)
{
  (void)sinkIndex;
  if(m_enSource_Current_Active != srcID)
  {
    m_enSource_Previous_Active = m_enSource_Current_Active;
    m_enSource_Current_Active = srcID;

    ETG_TRACE_USR4(("vSetCurrentActiveSource(): updated m_enSource_Current_Active, SourceClass: %d subID %d", srcID.enSourceClass, srcID.u16SubSource));
    ETG_TRACE_USR4(("vSetCurrentActiveSource(): updated m_enSource_Previous_Active, SourceClass: %d subID %d", m_enSource_Previous_Active.enSourceClass, m_enSource_Previous_Active.u16SubSource));
  }
}

/************************************************************************
 * FUNCTION     : vReadDataPool
 * DESCRIPTION  : function to read Last source from data pool
 * PARAMETER    : tU8& , tU16&
 * RETURNVALUE  : bool
 * HISTORY      :
 *              12.09.17 Vyankatesh VD  Initial Revision.
 ************************************************************************/
tVoid clAudioSMEngine::vReadDataPool(tU8& u8SrcID, tU16& u16SubSrcID)
{
  std::string SrcName;
  vReadDataPool(u8SrcID,u16SubSrcID,1,SrcName);
    ETG_TRACE_USR4(("vReadDataPool: Loaded Last source from data pool SourceClass: %d,subID: %d", u8SrcID, u16SubSrcID));
}

tVoid clAudioSMEngine::vReadDataPool(tU8& u8SrcID, tU16& u16SubSrcID,tU16 sinkID,std::string& SourcName)
{
  AudioUserDpIf* pAudUsrDpIf = AudioUserDpIfSelect::pGetAudUsrDpIf();
  std::string SrcName("\0");

  if(pAudUsrDpIf == NULL)
  {
    ETG_TRACE_FATAL(("Can not read Datapool"));
    return;
  }

  ETG_TRACE_USR4(("vReadDataPool for sinkID %d at index %d ",sinkID,clGeniviAudioCtrlAdapter::GetSinkIndex(sinkID)));

  // use default if read fails
  u8SrcID = clFactory_AudioSourceClass::GetDefaultSourceClass();
    SourcName = clFactory_AudioSourceClass::GetDefaultSourceName();
    u16SubSrcID = (tU16)0;

  if(pAudUsrDpIf->s32GetLastSourceClassOfSink(SrcName,(tU16)clGeniviAudioCtrlAdapter::GetSinkIndex(sinkID)) == 0)
	{
		ETG_TRACE_USR4(("vReadDataPool s32GetLastSourceClassOfSink gave 0 -> return"));
		return;
	}
  //convert source name to source id and subid
  if(!SrcName.empty())
  {
    ETG_TRACE_USR4(("vReadDataPool:%s",SrcName.c_str()));
    SourcName = SrcName;
    //saperate sourceid and subid
    if(SrcName.find("#")  != std::string::npos)
    {
      std::string subSrcName = SrcName.substr(SrcName.find("#")+1);
      tU16 subSrc = (tU16)atoi(subSrcName.c_str());
      SrcName = SrcName.substr(0,SrcName.find("#"));
      u16SubSrcID = subSrc;
    }
    ETG_TRACE_USR4(("vReadDataPool: Srcname %s",SrcName.c_str()));
    const clSourceClass* srcClass = clFactory_AudioSourceClass::GetSourceClass(SrcName.c_str());
    if(!srcClass) return;
    clAudioSource* src = clAudioSourceFactory::getAudioSource(srcClass->SourceClassID,0);
    if(!src)  return ;
    u8SrcID = src->sGetId().enSourceClass;
    ETG_TRACE_USR4(("vReadDataPool:%d",u8SrcID));
  }
  else
  {
      ETG_TRACE_USR4(("vReadDataPool:invalid source restore FM"));
  }
    ETG_TRACE_USR4(("vReadDataPool: Loaded Last source from data pool SourceClass: %d,subID: %d", u8SrcID, u16SubSrcID));
}

/************************************************************************
 * FUNCTION     : vWriteDataPool
 * DESCRIPTION  : function to update Last source into data pool
 * PARAMETER    : tVoid
 * RETURNVALUE  : tVoid
 * HISTORY      :
 *              12.09.17 Vyankatesh VD  Initial Revision.
 ************************************************************************/
tVoid clAudioSMEngine::vWriteDataPool(bool bUserProfileChanged)
{
    ETG_TRACE_USR4(("vWriteDataPool: entered bUserProfileChanged = %d",bUserProfileChanged));
    //ETG_TRACE_USR4(("!! DISABLED !!!"));
  //clAudioSource* poAudioSource = clAudioSourceFactory::getAudioSource(u8SrcID);
    //vWriteDataPool(m_enSource_Entertainment_Active[clGeniviAudioCtrlAdapter::GetSinkIndex(sinkID)].enSourceClass,m_enSource_Entertainment_Active.u16SubSource,bUserProfileChanged);
}

/************************************************************************
 * FUNCTION     : vWriteDataPool
 * DESCRIPTION  : function to update Last source into data pool
 * PARAMETER    : tU8& , tU16&
 * RETURNVALUE  : tVoid
 * HISTORY      :
 *              12.09.17 Vyankatesh VD  Initial Revision.
 ************************************************************************/
/*tVoid clAudioSMEngine::vWriteDataPool(tU8& u8SrcID, tU16& u16SubSrcID, bool bUserProfileChanged)
{
    ETG_TRACE_USR4(("vWriteDataPool: entered, SourceClass: %d,subID: %d",u8SrcID,u16SubSrcID));
  AudioUserDpIf* pAudUsrDpIf = AudioUserDpIfSelect::pGetAudUsrDpIf();
  if(pAudUsrDpIf == NULL)
  {
    ETG_TRACE_FATAL(("Can not access Datapool"));
    return;
  }
  ETG_TRACE_FATAL(("vWriteDataPool DP access O.K. for Src %d, SubSrc %d and Sink =1 but currently disabled",u8SrcID,u16SubSrcID));

    //convert sourceID to Source name
  clAudioSource* poAudioSource = clAudioSourceFactory::getAudioSource(u8SrcID);
  if(!poAudioSource) return;
  std::string subId = std::to_string(u16SubSrcID);
  std::string SrcName(poAudioSource->pacGetName());
  vWriteDataPool(u8SrcID,u16SubSrcID,clFactory_AudioSourceClass::GetDefaultSinkID());
  //pAudUsrDpIf->s32SetLastSourceClassOfSink(u8SrcID,1);
  //pAudUsrDpIf->s32SetLastSourceSubIDOfSink(u16SubSrcID,1);


#if 0
    if (bLoadUsrProfileDataPool())
    {
        ETG_TRACE_USR1(("vWriteDataPool: Updating last source to multi user data pool for Personalization"));

        dp_tclAudioStackUsrDPAudioStack_Usr_LastSource_Class oLastSource_Usr_Class;
        oLastSource_Usr_Class.vSetData(u8SrcID);

        dp_tclAudioStackUsrDPAudioStack_Usr_LastSource_SubID oLastSource_Usr_SubID;
        oLastSource_Usr_SubID.vSetData(u16SubSrcID);
    }
    else if(!bUserProfileChanged)
    {
        ETG_TRACE_USR1(("vWriteDataPool: Updating data pool with last source"));

        dp_tclAudioStackDPAudioStack_LastSource_Class oLastSource_Class;
        oLastSource_Class.vSetData(u8SrcID);

        dp_tclAudioStackDPAudioStack_LastSource_SubID oLastSource_SubID;
        oLastSource_SubID.vSetData(u16SubSrcID);
    }
#endif
}
*/
am_sourceID_t clAudioSMEngine::getSourceIDbyName(std::string srcName)
{
  std::vector<am_Source_s> listSources;
  //am_Error_e err = clGeniviAudioCtrlAdapter::GetGeniviCtrlIF()->getListSources(listSources);
  (void)clGeniviAudioCtrlAdapter::GetGeniviCtrlIF()->getListSources(listSources);

  am_sourceID_t sourceID =0;

  //Now search through list and find the name
  for(unsigned int i=0;i<listSources.size();i++)
  {
    if(listSources[i].name==srcName)
    {
      //we have it
      sourceID=listSources[i].sourceID;
      break;
    }
  }

  return sourceID;
}

am_sinkID_t clAudioSMEngine::getSinkIDbyName(std::string sinkName)
{
  std::vector<am_Sink_s> listSinks;
  //am_Error_e err = clGeniviAudioCtrlAdapter::GetGeniviCtrlIF()->getListSinks(listSinks);
  (void)clGeniviAudioCtrlAdapter::GetGeniviCtrlIF()->getListSinks(listSinks);

  am_sinkID_t sinkID =0;

  //Now search through list and find the name
  for(unsigned int i=0;i<listSinks.size();i++)
  {
    if(listSinks[i].name==sinkName)
    {
      //we have it
      sinkID=listSinks[i].sinkID;
      break;
    }
  }

  return sinkID;
}

void clAudioSMEngine::printConnectionList()
{
  std::vector<am_Connection_s> listConnections;
  am_Error_e err = clGeniviAudioCtrlAdapter::GetGeniviCtrlIF()->getListConnections(listConnections);
  ETG_TRACE_USR4(("clAudioSMEngine::printConnectionList(), err %d",err));
  ETG_TRACE_USR4(("----------------------------------------------------------------"));
  for(unsigned int i=0;i<listConnections.size();i++)
  {
    ETG_TRACE_USR4(("ConnectionID %d: SourceID %d -> SinkID %d",listConnections[i].connectionID,listConnections[i].sourceID,listConnections[i].sinkID));
  }
  ETG_TRACE_USR4(("----------------------------------------------------------------\n"));
}

void clAudioSMEngine::printSubConnection(am_connectionID_t connectionID)
{
  std::vector<am_Connection_s> listConnections;
  am_Error_e err = clGeniviAudioCtrlAdapter::GetGeniviCtrlIF()->getListConnections(listConnections);
  ETG_TRACE_USR4(("clAudioSMEngine::printConnectionList() for connectionID %d, err=%d",connectionID,err));
  ETG_TRACE_USR4(("----------------------------------------------------------------"));
  for(unsigned int i=0;i<listConnections.size();i++)
  {
    if(listConnections[i].connectionID == connectionID)
    {
      ETG_TRACE_USR4(("\t ConnectionID %d: SourceID %d -> SinkID %d",listConnections[i].connectionID,listConnections[i].sourceID,listConnections[i].sinkID));
      break;
    }
  }
  ETG_TRACE_USR4(("----------------------------------------------------------------\n"));

}

void clAudioSMEngine::printMainConnectionList()
{
  std::vector<am_MainConnection_s> listMainConnections;
  am_Error_e err = clGeniviAudioCtrlAdapter::GetGeniviCtrlIF()->getListMainConnections(listMainConnections);
  ETG_TRACE_USR4(("clAudioSMEngine::printMainConnectionList(), err=%d",err));
  ETG_TRACE_USR4(("----------------------------------------------------------------"));
  for(unsigned int i=0;i<listMainConnections.size();i++)
  {
    ETG_TRACE_USR4(("MainConnectionID %d: SourceID %d -> SinkID %d",
        listMainConnections[i].mainConnectionID,listMainConnections[i].sourceID,listMainConnections[i].sinkID));
    //print list of sub
    for(unsigned int k=0;k<listMainConnections[i].listConnectionID.size();k++)
    {
      printSubConnection(listMainConnections[i].listConnectionID[k]);
    }
  }
  ETG_TRACE_USR4(("----------------------------------------------------------------\n"));
}

void clAudioSMEngine::vWriteDPexternalSink(std::string srcName, tU16 sinkID)
{
  ETG_TRACE_USR4(("writeDPexternalSink: entered, sinkID %d, srcName %s",sinkID,srcName.c_str()));
  AudioUserDpIf* pAudUsrDpIf = AudioUserDpIfSelect::pGetAudUsrDpIf();
  if(pAudUsrDpIf == NULL)
  {
    ETG_TRACE_FATAL(("Can not access Datapool"));
    return;
  }
  int sink = clGeniviAudioCtrlAdapter::GetSinkIndex(sinkID);
  if(sink > -1)
  {

    if(srcName.find("#")  != srcName.npos)  //if # is not the last character (something will follow)
    {
      tU16 u16SubSrcID=0;

      size_t pos=srcName.find("#")+1;
      size_t len=srcName.size()-pos;

      ETG_TRACE_USR4(("vWriteDataPool: pos=%d,len=%d,npos = %d",(int)pos,(int)len,srcName.npos));

      std::string subSrcStr = srcName.substr(pos,len);

      ETG_TRACE_USR4(("vWriteDataPool: subSrcStr name is now %s",subSrcStr.c_str()));
      u16SubSrcID = (tU16)std::stoi(subSrcStr);

      srcName = srcName.substr(0,srcName.find("#"));
      ETG_TRACE_USR4(("vWriteDataPool: Src name is now %s",srcName.c_str()));

      ETG_TRACE_USR4(("vWriteDataPool: u16SubSrcID = %d, Src name %s",u16SubSrcID,srcName.c_str()));

      pAudUsrDpIf->s32SetLastSourceClassOfSink((tString)srcName.c_str(),(tU16)sink);
      pAudUsrDpIf->s32SetLastSourceSubIDOfSink(u16SubSrcID,(tU16)sink);
      ETG_TRACE_USR4(("vWriteDataPool: sink = %d, subId = %d, Src name: %s",sink, u16SubSrcID, srcName.c_str()));
    }
    else
    {
      pAudUsrDpIf->s32SetLastSourceClassOfSink((tString)srcName.c_str(),(tU16)sink);
      pAudUsrDpIf->s32SetLastSourceSubIDOfSink(0,(tU16)sink);
      ETG_TRACE_USR4(("vWriteDataPool: sink = %d, subId = 0, Src name: %s",sink, srcName.c_str()));
    }

  }
  else
  {
    ETG_TRACE_USR4(("vWriteDataPool: Error no valid sink %d from sinkID %d",sink,sinkID));
  }
}


tVoid clAudioSMEngine::vWriteDataPool(tU8& u8SrcID, tU16& u16SubSrcID, tU16 sinkID)
{
    ETG_TRACE_USR4(("vWriteDataPool: entered, SourceClass: %d,subID: %d, sinkID %d",u8SrcID,u16SubSrcID,sinkID));
  AudioUserDpIf* pAudUsrDpIf = AudioUserDpIfSelect::pGetAudUsrDpIf();
  if(pAudUsrDpIf == NULL)
  {
    ETG_TRACE_FATAL(("Can not access Datapool"));
    return;
  }
  //convert sourceID to Source name if it doesnot have valid source name .
  clAudioSource* poAudioSource = clAudioSourceFactory::getAudioSource(u8SrcID);
  if(!poAudioSource) return ;
  std::string subId = std::to_string(u16SubSrcID);
  std::string SrcName(poAudioSource->pacGetName());
  if(u8SrcID)
  {
    if(SrcName.find("#")  != std::string::npos)
    {
      SrcName = SrcName.substr(0,SrcName.find("#")+1);
      SrcName.append(std::to_string(u16SubSrcID));
      ETG_TRACE_USR4(("vWriteDataPool: Src name %s",SrcName.c_str()));
    }
    ETG_TRACE_USR4(("vWriteDataPool: Src name %s",SrcName.c_str()));
  }
  else
  {
    SrcName = clFactory_AudioSourceClass::GetDefaultSourceName();
  }

  //write to datapool at perticular sink
  int sink = clGeniviAudioCtrlAdapter::GetSinkIndex((am_sinkID_t)sinkID);

  if(sink <= -1) return ;

  //check if visible source
  std::vector<am_Source_s> listSources;
  if(clGeniviAudioCtrlAdapter::m_geniviControlIF->getListSources(listSources)== E_OK)
  {
    for (std::vector<am_Source_s>::iterator it = listSources.begin() ; it != listSources.end(); ++it)
    {
      if ((it->name).compare(SrcName)==0 && it->visible == true)
      {
      pAudUsrDpIf->s32SetLastSourceClassOfSink((tString)SrcName.c_str(),(tU16)sink); //ssw3kor
    ETG_TRACE_USR4(("vWriteDataPool: Src name %s",SrcName.c_str()));
      }
    }
  }


}




void clAudioSMEngine::SetDP()
{
  AudioUserDpIf* pAudUsrDpIf = AudioUserDpIfSelect::pGetAudUsrDpIf();

  for(unsigned int i=0; i<clGeniviAudioCtrlAdapter::m_sinkList.size();i++)
  {
    ETG_TRACE_USR4(("sinkID %d with sinkName %s",clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.sinkID,
        clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.name.c_str()));
    ETG_TRACE_USR4(("domainID %d",clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.domainID));
    ETG_TRACE_USR4(("visible %d",clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.visible));
    if(clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.domainID == clGeniviAudioCtrlAdapter::GetDomainID())
    {
      ETG_TRACE_USR4(("set TUNER_FM for sinkID %d, (i=%d)",clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.sinkID,i));
      pAudUsrDpIf->s32SetLastSourceClassOfSink((tString)"TUNER_FM",(tU16)i); //ssw3kor
    }
    else
    {
      ETG_TRACE_USR4(("clear DP for sinkID %d, (i=%d)",clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.sinkID,i));
      pAudUsrDpIf->s32SetLastSourceClassOfSink((tString)"",(tU16)i); //ssw3kor
    }
  }
}


void clAudioSMEngine::SetDPSubIDForSinkID(tU16 sinkID, tU16 subID)
{
  AudioUserDpIf* pAudUsrDpIf = AudioUserDpIfSelect::pGetAudUsrDpIf();

  for(unsigned int i=0; i<clGeniviAudioCtrlAdapter::m_sinkList.size();i++)
  {
    ETG_TRACE_USR4(("sinkID %d with sinkName %s",clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.sinkID,
        clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.name.c_str()));
    ETG_TRACE_USR4(("domainID %d",clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.domainID));
    ETG_TRACE_USR4(("visible %d",clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.visible));
    if(clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.sinkID == sinkID)
    {
      ETG_TRACE_USR4(("set subID %d for sinkID %d, (i=%d)",subID,clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.sinkID,i));
      pAudUsrDpIf->s32SetLastSourceSubIDOfSink(subID,(tU16)i); //ssw3kor
    }
    else
    {
      ETG_TRACE_USR4(("ignore sinkID %d, (i=%d)",clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.sinkID,i));
    }
  }
}

void clAudioSMEngine::SetDPForSinkID(tU16 sinkID, tU16 sourceID)
{

  am_Source_s sourceData;
  sourceData.sourceID = 0;
  sourceData.domainID=0;
  sourceData.sourceClassID = 0;
  sourceData.sourceState = SS_OFF;
  sourceData.visible = false;
  sourceData.volume = 0;

  am_SourceClass_s classInfo;

  am_Error_e err = clGeniviAudioCtrlAdapter::GetGeniviCtrlIF()->getSourceInfoDB(sourceID,sourceData);
  if(err != E_OK)
  {
	  ETG_TRACE_USR4(("SetDPForSinkID: getSourceInfoDB for sourceID %d gave err %d",sourceID,err));
  }
  err = clGeniviAudioCtrlAdapter::GetGeniviCtrlIF()->getSourceClassInfoDB(sourceID, classInfo);
  if(err != E_OK)
  {
	  ETG_TRACE_USR4(("SetDPForSinkID: getSourceClassInfoDB for sourceID %d gave err %d",sourceID,err));
  }
  AudioUserDpIf* pAudUsrDpIf = AudioUserDpIfSelect::pGetAudUsrDpIf();
  if(pAudUsrDpIf == NULL)
  {
	  ETG_TRACE_ERR(("SetDPForSinkID: pGetAudUsrDpIf gave NULL can not continue"));
	  return;
  }

  for(unsigned int i=0; i<clGeniviAudioCtrlAdapter::m_sinkList.size();i++)
  {
    ETG_TRACE_USR4(("sinkID %d with sinkName %s",clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.sinkID,
        clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.name.c_str()));
    ETG_TRACE_USR4(("domainID %d",clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.domainID));
    ETG_TRACE_USR4(("visible %d",clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.visible));
    if(clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.sinkID == sinkID)
    {
      ETG_TRACE_USR4(("set %s",sourceData.name.c_str()));
      ETG_TRACE_USR4(("for sinkID %d, (i=%d)",clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.sinkID,i));
      pAudUsrDpIf->s32SetLastSourceClassOfSink((tString)sourceData.name.c_str(),(tU16)i); //ssw3kor
    }
    else
    {
      ETG_TRACE_USR4(("ignore sinkID %d, (i=%d)",clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.sinkID,i));
    }
  }

}

void clAudioSMEngine::ClearDPForSinkID(tU16 sinkID)
{
  AudioUserDpIf* pAudUsrDpIf = AudioUserDpIfSelect::pGetAudUsrDpIf();

  for(unsigned int i=0; i<clGeniviAudioCtrlAdapter::m_sinkList.size();i++)
  {
    ETG_TRACE_USR4(("sinkID %d with sinkName %s",clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.sinkID,
        clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.name.c_str()));
    ETG_TRACE_USR4(("domainID %d",clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.domainID));
    ETG_TRACE_USR4(("visible %d",clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.visible));
    ETG_TRACE_USR4(("\n"));
    if(clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.sinkID == sinkID)
    {
      ETG_TRACE_USR4(("clear DP for sinkID %d, (i=%d)",clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.sinkID,i));
      pAudUsrDpIf->s32SetLastSourceClassOfSink((tString)"",(tU16)i); //ssw3kor
    }
    else
    {
      ETG_TRACE_USR4(("ignore sinkID %d, (i=%d)",clGeniviAudioCtrlAdapter::m_sinkList[i].s_sink.sinkID,i));
      ETG_TRACE_USR4(("\n"));
    }
  }
}

/*************************************************************************
* METHOD:         vDPPersistentMute
*
* DESCRIPTION:    Sets or Resets the persistenet mute based on some conditions
*
* PARAMETER:      poAudioSource: only if it is MUTE_ENTERTAIN, else no action
*                 action: set or reset for PersMute
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vDPPersistentMute(AudioSource::clAudioSource* poAudioSource, enPersMuteDPAction action)
{
  ETG_TRACE_USR3(("vDPPersistentMute():Entered in vDPPersistentMute"));
  //Check if Diag Session is active
  if(true == clAudioSourceController::getInstance().bIsTopOfBackUpStackMuteAndOn())
  {
    ETG_TRACE_USR4(("vDPPersistentMute():Diagnosis session is active, No need to update Persistent Mute in datapool !"));
    return;
  }

  if(0 == OSAL_s32StringCompare(poAudioSource->pacGetName(),"MUTE_ENTERTAIN"))
  {

    //TODO change this to below
//  AudioUserDpIf* pAudUsrDpIf = AudioUserDpIfSelect::pGetAudUsrDpIf();
//  if(pAudUsrDpIf == NULL)
//  {
//    ETG_TRACE_FATAL(("Can not access Datapool"));
//    return;
//  }
//
//  pAudUsrDpIf->

    //we store this persistent
    //dp_tclAudioStackDPAudioStack_PersistentMute   oPersMute;
    dp_tclAudioStackMultiUsrDPAudioStackMulti_PersistentMute oPersMute;
    if(action == PersMuteDP_Reset)
    {
      ETG_TRACE_USR3(("Reset Persistent Mute source MUTE_ENTERTAIN"));
      (void)oPersMute.s32SetData(0);

#if defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_RN_AIVI)
      ETG_TRACE_USR4(("vDPPersistentMute need to update SystemProperty"));
      clGeniviAudioCtrlAdapter::updateSystemPropertyDB(SYP_USER_MUTE_STATUS,SYP_USER_MUTE_STATUS_OFF);
#endif
    }
    else
    {
      // to avoid persistency of non user action MUTE_ENTERTAIN during shutdown
      // check if MUTE_SYSTEM is on stack and if active the MUTE_ENTERTAIN was not triggerd by the user
      const clSourceClass* pSrcClass_SysMute = clFactory_AudioSourceClass::GetSourceClass("MUTE_SYSTEM");
      if(pSrcClass_SysMute==NULL)
      {
        ETG_TRACE_ERR(("Audio Source Class MUTE_SYSTEM not found"));
        return;
      }
      if(clGeniviAudioCtrlAdapter::bIsCurrentActiveAudiosource(SourceID(pSrcClass_SysMute->getClassID(),0)))
      {
        ETG_TRACE_USR3(("MUTE_SYSTEM is active -> MUTE_ENTERTAIN not triggered by user, no persistent storage"));
      }
      else
      {
        ETG_TRACE_USR3(("MUTE_SYSTEM not active > MUTE_ENTERTAIN triggered by user, persistent storage"));
        (void)oPersMute.s32SetData(1);
      }
    }
  }
}
}//namespace
