/*
 * clConfiguration.h
 *
 *  Created on: Jan 9, 2014
 *      Author: vo84hi
 */

#ifndef AUDIOSTACK_AUDIOSOURCECLASS_H_
#define AUDIOSTACK_AUDIOSOURCECLASS_H_

#include "AudioStack/InterfaceAudioStack.h"

#include "AudioStack/clGateway.h"

#include "AudioStack/clStackRules.h"
#include "AudioStack/AudioSources/clAudioSource.h"

//includes for loading XML ruleset
#include <string>
#include <vector>
#include <libxml/parser.h>
#include <libxml/tree.h>

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_audio_if.h"

#define INTERNALAMP_2SPEAKER 0
#define PREMIUM_AMP1 1
#define EXTERNALAMP_BOSEWOMCAN 2
#define PREMIUM_AMP2 3
#define EXTERNALAMP_BOSE2CH_MCAN 4
#define EXTERNAL_ANC 5
#define EXTERNALAMP_BOSE_5_1CH_MCAN 6
#define EXTERNAL_DAIMLER_1 7
#define HEAD_UNIT_ALONE 8
#define INTERNALAMP_4SPEAKER 9
#define EXTERNAL_DAIMLER_2 0x0A

namespace AudioStack
{
namespace AudioSource
{

class clAudioSourceController;
/**
 * @Brief
 * This class provides functionality to load
 * sources and their attributes at runtime
 * Methods and members are gernerally static, so no
 * instance of the class is needed.
 */
class clFactory_AudioSourceClass
{

public:
   //inline
   static tVoid vSetAudioSourceControllerCallback(clAudioSourceController* poAudioSourceControllerCallback){m_pAudioSourceController=poAudioSourceControllerCallback;}

   static tVoid Invalidate();
   /**
    * Check if Source is valid.
    * It returns true if a corresponding StackRule is found.
    */
   static tBool isValidSourceClass(sourceClassID srcClass);
   /**
    * Get the Group of a given SourceID
    * @return TRUE if a group is found. FALSE if there is no matching Group
    */
   static tU16  GetAvailabilityTimeout(SourceID& srcID);
   /**
    * Get the Group of a given SourceID
    * @return TRUE if a group is found. FALSE if there is no matching Group
    */
   static clSourceClass::sourceGroupID  GetGroupID          (SourceID srcID);
   /**
    * Supplies TypeID of a SourceClass. The Types can be defined in the
    * XML.
    */
   static clSourceClass::sourceTypeID   GetTypeID           (SourceID srcID);

   static const stSourceType& GetType(SourceID srcID);

   static const stSourceSetting& GetSourceSettings()
   {
     return m_source_settings;
   }

   /**
    * Returns the registration behavior of a SourceClass.
    * This is defined and loaded from Rule XML.
    * @return How the source is registered to Genivi AudioManager
    */
   static clSourceClass::registration_t GetRegistrationMode (SourceID srcID);
   /**
    * Get the AutoPlay configuration of a Source Class
    * This is defined and loaded from Rule XML.
    * @return TRUE if source is started automatically if Source availability
    * changes to Playable (e.g. via ARL SetSourceAvailability)
    */
   static clSourceClass::autoplay_t     GetAutoPlayMode     (SourceID srcID);
   /**
    * Retrieve the externalID of a SourceClass, e.g. the CCA Source value
    * This is defined and loaded from Rule XML.
    */
   static tU32                          GetExternalID       (SourceID srcID);
   /**
    * Will return the SourceClass that is marked as default source class
    * within StackRules.
    */
   static sourceClassID                 GetDefaultSourceClass();
   static std::string                   GetDefaultSourceName();
   static tU16                          GetDefaultSinkID();

   /**
    * Get the mandatory, implicitly and always available
    * sourceClass (typically ID=0) which can be used for
    * UNDEF or INVALID SourceClass.
    * This can be used to check if the SourceClass is not initializied
    * or marked as invalid.
    * @return ID of a sourceClass that is INVALID.
    */
   static sourceClassID                 GetInvalidSourceClass();

   /**
    * Get a source class by external id
    * returns NULL if no sourceClass is found
    */
   static const clSourceClass* GetSourceClass_extID(clSourceClass::externalID sxtID) ;

   /**
    * Get a source class by name
    * returns NULL if no sourceClass is found
    */
   static const clSourceClass* GetSourceClass(const char * srcClassName) ;

   /**
    * Get a SourceClass reference
    */
   static const clSourceClass& GetSourceClass(sourceClassID srcClass);

   /**
    * Parse and print the XML Ruleset
    */
   static tVoid ReadXML                   ();
   /**
    * Print the actual Ruleset, that has been loaded in prior
    */
   static tVoid Ruleset_Print             ();
   /**
    * Load XML initial Startpoint
    */
   static tBool Ruleset_Load              ();
   static tBool Ruleset_Load(std::string xmlFileint, int sinkNr);

   /**
    * Set the ruleset to other than default
    */
   static tVoid SetRulesetFile(const char *filename);

   static bool bIsMCanBoseAmplifierConnected();


   static const std::vector<stSourceType>& GetSourceTypeList()
   {
      return m_source_types;
   }

   static void SetDynamicSinkID(am_sinkID_t sinkID, const am_Sink_s& gam_sink_s);
   static void SetDynamicSinkIDSourceClasses(am_sinkID_t sinkID, std::string name);

   static void SetDynamicSourceID(am_sourceID_t sourceID, const am_Source_s& gam_source_s);

   static bool bIsGatewayRegistered(am_domainID_t domainID);

   static std::vector<clSourceClass>& GetSourceClassListNoConst();
   static const std::vector<clSourceClass>* GetSourceClassListPtr();
   static const std::vector<clSourceClass>& GetSourceClassList()
   {
      return m_source_class;
   }

   //static const std::vector<std::pair<tU16,std::string>>& GetSystemSinkList()
   static const std::vector<std::pair<am_sinkID_t, am_Sink_s>>& GetSystemSinkList()
   {
       return m_systemSinks;
   }
/***********************************************************************/
/*                            PRIVATE SECTION                          */
/***********************************************************************/
private:
   static tVoid Ruleset_Parse             ();
   static tVoid Ruleset_Parse_Print       (xmlNode * a_node, int depth );
   /**
    * Load Types
    * @return TRUE loading completed successfully, FALSE on error
    */
   static tBool Ruleset_LoadTypes         (xmlNode * a_node);

   /**
    * Load settings
    * @return TRUE loading completed successfully, FALSE on error
    */
  static tBool Ruleset_LoadSettings        (xmlNode * a_node);

   /**
    * Load groups
    * @return TRUE loading completed successfully, FALSE on error
    */
   static tBool Ruleset_LoadGroups        (xmlNode * a_node);
   /**
    * Load sinks (all possible sinks)
    * @return TRUE loading completed successfully, FALSE on error
    */
   static tBool Ruleset_LoadSinks        (xmlNode * a_node);

   /**
   * Load allowed sinks for a source
   * @return TRUE loading completed successfully, FALSE on error
   */
   static tBool Ruleset_LoadSource_AllowedSinks(xmlNode * sinkGroupNode);

   /**
    * Iterate SourceNodes and initiate loading of sourceNodes
    * @return TRUE loading completed successfully, FALSE on error
    */
   static tBool Ruleset_LoadSources       (xmlNode * a_node);
   /**
    * Inspection and delegate SourceNode children
    * @return TRUE loading completed successfully, FALSE on error
    */
   static tBool Ruleset_LoadSource_SinkRuleAndAction   (xmlNode * sourceNode);
   static tBool Ruleset_LoadSource_Sink   (xmlNode * sourceNode);

   static tBool Ruleset_LoadSource_RuleAndAction   (xmlNode * sourceNode);
   /**
    * Load Source nodes from XML
    * @return TRUE loading completed successfully, FALSE on error
    */
   static tBool Ruleset_LoadSource        (xmlNode * a_node);
   /**
    * Load Rules of a SourceNode
    * @return TRUE loading completed successfully, FALSE on error
    */
   static tBool Ruleset_LoadSource_Rules  (xmlNode * rulesGroupNode);
   /**
    * Load Actions of a SourceNode
    * @return TRUE loading completed successfully, FALSE on error
    */
   static tBool Ruleset_LoadSource_Actions(xmlNode * rulesGroupNode);
   /**
    * assigns sourceClassID to given id parameter
    * @return TRUE completed successfully, FALSE on error
    */
   static tBool Ruleset_GetSourceClassID  (sourceClassID &id, const char * sourceClassName);


   static tBool Ruleset_LoadGateways(xmlNode* a_node);
   static tBool Ruleset_LoadGatewaySubNodes(xmlNode* gwNode, am_Gateway_s& aGateway, clGateway* pclGateway);

   static tBool Ruleset_LoadGatewaySubNodesLists(xmlNode* gwNode,am_Gateway_s& aGateway,clGateway* pclGateway);

   static tBool Ruleset_LoadGW_SourceFormats(xmlNode* srcFormatNode, am_Gateway_s& aGateway);
   static tBool Ruleset_LoadGW_SinkFormats(xmlNode* sinkFormatNode, am_Gateway_s& aGateway);
   static tBool Ruleset_LoadGW_ConversionMatrix(xmlNode* ConvMatrixNode, am_Gateway_s& aGateway);


/***********************************************************************/
/*                            MEMBER  SECTION                          */
/***********************************************************************/
   static std::string                m_configFile;
   static std::string                m_configBoseFile;
   static std::vector<stSourceType>  m_source_types;
   static stSourceSetting          m_source_settings;
   static std::vector<std::string>   m_source_groups;
   static std::vector<clSourceClass> m_source_class;

   static clAudioSourceController* m_pAudioSourceController;
   // known sinks in the system
   static std::vector<tU16> m_sinks;
   //static std::vector<std::pair<tU16,std::string>> m_systemSinks;  //all sinks that are known in the system (including dynamic sinks with unknown ID 0)
   static std::vector<std::pair<am_sinkID_t, am_Sink_s>> m_systemSinks;  //all sinks that are known in the system (including dynamic sinks with unknown ID 0)


   //ToDo 27.07.2016 daw2hi we should use sink class instead
   //static std::vector<clSinkClass> m_sink_class;

   static std::vector<am_Gateway_s> m_gateways;  //added 14.02.1018

   //it seems we need a class here to achieve gateway registration when all information is available
   static std::vector<clGateway> m_clGateways;  //added 30.08.2018

   static sourceClassID              m_default_sourceClassId;
   static tU16                       m_default_sinkID;
   static std::string                m_default_sourcename;
   static long                       m_timeXmlLoad_overallTime;
   static long                       m_timeXmlLoad_cpuTime;
};

}}


#endif /* CLCONFIGURATION_H_ */
