/*!************************************************************************
 * FILE :         clAudioSource.h
 * SW-COMPONENT:  Audio Engine
 * DESCRIPTION:
 *
 * Each AudioSource in the system is represented by an instance of AudioSource
 * or by a subclass of AudioSource.
 * AudioSource instances are used and controlled by the AudioSourceController.
 *
 * The individual subclasses of AudioSources implement the source-specific
 * communication with the corresponding MW components.
 *
 * Each AudioSource exposes an interface to the AudioSourceController with
 * the methods:
 * - vOn(...)     start source (player) and start demute-ramp
 * - vOff(...)    start mute-ramp and stop source (player)
 * - vPause(...)  start mute-ramp and pause source (player)
 * - bIsMuted(...)
 * - vReset(...)  stop player and mute source
 *
 * The AudioSource notifies the AudioSourceController via Callback-methods:
 * - vOn_Done()    the demute-ramp was started
 * - vOff_Done()...the mute-ramp has finished, player was stopped
 * - vPause_Done() the mute-ramp has finished, player was paused
 *
 * The AudioSource can have the states: (please see the state-charts for details)
 * - Off
 * - Pause
 * - On        demute-ramp was finished. Source is playing
 * - RampUp    AudioSource has started the demute-ramp
 * - RampDown  AudioSource has started the mute-ramp
 *
 * A statemachine ensures that requests are performed following the state-
 * transition rules.
 *
 * Sequence is:
 * - AudioSourceController calls vOn, vOff or vPause
 * - clAudioSource delegates request to StateMachine
 * - StateMachine applies rule-checks and might invoke vMW_On, vMW_Off, vMW_Pause
 * - vMW_On, vMW_Off, vMW_Pause do the communication with the MW
 *   Individual SubClasses of clAudioSouce may overwrite this methods to
 *   establish a specific MW communcation.
 * - Each implementation of vMW_On, vMW_Off, vMW_Pause has to confirm the operation
 *   asynchronously by calling vMW_OnDone, vMW_OffDone, vMW_PauseDone
 * - vMW_OnDone, vMW_OffDone, vMW_PauseDone will delegate the trigger
 *   again to the StateMachine.
 * - The StateMachine applies again the rules and invokes one of the methods:
 *   vNotifyOnDone,  vNotifyOffDone and  vNotifyPauseDone
 *   in order to inform the AudioSourceController
 *
 * TASKS TO IMPLEMENT A NEW AUDIOSOURCE:
 * - Provide a subclass of clAudioSource
 * - Overwrite the methods: vMW_On, vMW_Off, vMW_Pause
 *   Implement within these methods the specific communication with MW
 * - Ensure that these methods confirm the operation by invoking:
 *   vMW_OnDone, vMW_OffDone, vMW_PauseDone
 *
 *
 * AUTHOR:        CM-DI/PJ-VW34 Steinle
 * COPYRIGHT:     (c) 2006 Blaupunkt Werke
 *************************************************************************/
#ifndef _AUDIOSOURCE_H_
#define _AUDIOSOURCE_H_

#include "AudioStack/InterfaceAudioStack.h"
#include "AudioStack/clStackRules.h"

#include "CAmSocketHandler.h"

#include <map>

namespace AudioStack
{

//Forward declare different Namespace class
namespace SourceStateMachine
{
   class clSrcState;
}
class clAudioSMEngine;

namespace AudioSource
{
class clAudioSource;



struct SourceID
{
   sourceClassID  enSourceClass;
   tU16           u16SubSource;
   SourceID(sourceClassID srcClass, tU16 subSource) :enSourceClass(srcClass),u16SubSource(subSource){}
   SourceID(const SourceID& srcID) : enSourceClass(srcID.enSourceClass),u16SubSource(srcID.u16SubSource){}
   //Avoid this constructor, but we need it because of stl map operator[]
   SourceID() : enSourceClass(static_cast<sourceClassID>(0)), u16SubSource(0) {}
   bool operator<( const SourceID & rhs ) const
   {
      return (
            ((((tU32)enSourceClass  )<< 16) | (tU32)  u16SubSource) <
            ((((tU32)rhs.enSourceClass)<< 16) | (tU32)rhs.u16SubSource) );
   }

   bool operator==(const SourceID& rhs) const
            {
      return (
            (rhs.enSourceClass == enSourceClass) &&
            (rhs.u16SubSource == u16SubSource) );
            }

   bool operator!=(const SourceID& rhs) const
               {
      return (
            (rhs.enSourceClass != enSourceClass) ||
            (rhs.u16SubSource != u16SubSource) );
               }

   SourceID& operator =(const SourceID& o)
   {
      enSourceClass = (o.enSourceClass);
      u16SubSource = (o.u16SubSource);
      return *this;
   }
};

class AudioStates{
public:
    typedef enum {
      UNDEF          = 0,
      OFF            = 1,
      ON             = 2,
      PAUSE          = 3,
      RAMPDOWN2OFF   = 4,
      RAMPDOWN2PAUSE = 5,
      RAMPUP         = 6,
      RAMPUPINIT     = 7,
      RAMPUPREQUEST  = 8,
      WAIT_FOR       = 9,
      SRC_AVAILABLE  = 10,
    START_MUTE     = 11,
    START_UN_MUTE  = 12
   } enAudioStates;
};

/**
 * \brief
 * clAudioSourceObserver provides an abstract interface to identify
 * an acknowledgement for source switch sequences
 */
class clAudioSourceObserver
{
public:
   virtual ~clAudioSourceObserver(){};
   /**
   * vOn_done is called from the corresponding source implementation or base class,
   * when the ON-command has been acknowledged by the midlleware
   */
   virtual tVoid vOn_done(clAudioSource*,tU16 SinkID)=0;
   /**
   * vOff_done is called from the corresponding source implementation or base class,
   * when the OFF-command has been acknowledged by the midlleware
   */
   virtual tVoid vOff_done(clAudioSource*,tU16 SinkID)=0;
   /**
   * vPause_done is called from the corresponding source implementation or base class,
   * when the PAUSE-command has been acknowledged by the midlleware
   */
   virtual tVoid vPause_done(clAudioSource*,tU16 SinkID)=0;
   /**
   * vInit_started is called from the corresponding source implementation or base class,
   * when the an initialisation sequence has been started
   */
   virtual tVoid vInit_started(clAudioSource*,tU16 SinkID)=0;

   virtual tVoid vStartMute(clAudioSource*,tU16 SinkID)=0;
   virtual tVoid vStartUnMute(clAudioSource*,tU16 SinkID)=0;
};

/**
 * \brief
 * Each AudioSource in the system is represented by an instance of AudioSource
 * or by a subclass of AudioSource.
 * AudioSource instances are used and controlled by the AudioSourceController.
 *
 * The individual subclasses of AudioSources implement the source-specific
 * communication with the corresponding MW components.
 *
 * Each AudioSource exposes an interface to the AudioSourceController with
 * the methods:
 * - vOn(...)     start source (player) and start demute-ramp
 * - vOff(...)    start mute-ramp and stop source (player)
 * - vPause(...)  start mute-ramp and pause source (player)
 * - bIsMuted(...)
 * - vReset(...)  stop player and mute source
 *
 * The AudioSource notifies the AudioSourceController via Callback-methods:
 * - vOn_Done()    the demute-ramp was started
 * - vOff_Done()...the mute-ramp has finished, player was stopped
 * - vPause_Done() the mute-ramp has finished, player was paused
 *
 * The AudioSource can have the states: (please see the state-charts for details)
 * - Off
 * - Pause
 * - On        demute-ramp was finished. Source is playing
 * - RampUp    AudioSource has started the demute-ramp
 * - RampDown  AudioSource has started the mute-ramp
 *
 * A statemachine ensures that requests are performed following the state-
 * transition rules.
 *
 * Sequence is:
 * - AudioSourceController calls vOn, vOff or vPause
 * - clAudioSource delegates request to StateMachine
 * - StateMachine applies rule-checks and might invoke vMW_On, vMW_Off, vMW_Pause
 * - vMW_On, vMW_Off, vMW_Pause do the communication with the MW
 *   Individual SubClasses of clAudioSouce may overwrite this methods to
 *   establish a specific MW communcation.
 * - Each implementation of vMW_On, vMW_Off, vMW_Pause has to confirm the operation
 *   asynchronously by calling vMW_OnDone, vMW_OffDone, vMW_PauseDone
 * - vMW_OnDone, vMW_OffDone, vMW_PauseDone will delegate the trigger
 *   again to the StateMachine.
 * - The StateMachine applies again the rules and invokes one of the methods:
 *   vNotifyOnDone,  vNotifyOffDone and  vNotifyPauseDone
 *   in order to inform the AudioSourceController
 *
 * TASKS TO IMPLEMENT A NEW AUDIOSOURCE:
 * - Provide a subclass of clAudioSource
 * - Overwrite the methods: vMW_On, vMW_Off, vMW_Pause
 *   Implement within these methods the specific communication with MW
 * - Ensure that these methods confirm the operation by invoking:
 *   vMW_OnDone, vMW_OffDone, vMW_PauseDone
 */
class clAudioSource : public am::IAmShTimerCallBack
{
public:

   typedef enum{
      On,
      Pause,
      Off
   } enSourceActivity;

   typedef enum {
      not_available
      , available
      , unkonwn
   } enSourceAvailability;

   typedef enum {
      newmedia
      , samemedia
      , nomedia
      , temperature
      , voltage
      , error
      , no_content
    , overcurrent
    , internaldisconnect
   } enSourceAvailabilityReason;

   friend class SourceStateMachine::clSrcState;
   friend class clAudioSourceFactory;

   virtual ~clAudioSource();
   virtual tBool bCCACallback(tU32 u32msgcode, tU32 u32msgparam1, tU32 msgparam2);
   clAudioSource(SourceID enSrcId);

   virtual tVoid vMW_CCAResponse(tS32 extSourceId, tU16 subSrc, enSourceActivity activity);
   virtual tVoid vMW_CCAResponse(tS32 extSrcID, tU16 subSrc, enSourceActivity activity, tU16 sinkID);

   virtual tVoid vCbAckDisconnect(const am_Handle_s*) {return;} //this is not doing anything in the base
   virtual tVoid vCbAckConnect(const am_Handle_s*) {return;}  //this is not doing anything in the base
   virtual tVoid vCbAckSetSourceState(const am_Handle_s*) {return;} //this is not doing anything in the base
   virtual void cbAckSetVolumeChange(const am_Handle_s* ,tU16 ){}

   /**
    * save the pointer to the object, which wants to observe the audio source switch sequence
    */
   tVoid vSetObserver(clAudioSourceObserver* poCallback){m_pclAudioSourceObserver=poCallback;}

   /**
    * availability of source changed
    */
   virtual tVoid vSourceAvailablilityChange(enSourceAvailability availability, enSourceAvailabilityReason availabilityReason);
   virtual tVoid vSourceAvailablilityChange();
   virtual tVoid vSourceAvailablilityChange(tU16 u16SinkID);

   tVoid vOn();
   tVoid vOff    (SourceID u8PossibleNextSource);
   tVoid vOff    ();
   tVoid vPause  (SourceID u8PossibleNextSource);
   tVoid vWaitFor();

   tVoid vOn(tU16 u16SinkID);
   tVoid vOff    (SourceID u8PossibleNextSource, tU16 u16SinkID);
   tVoid vOff    (tU16 u16SinkID);
   tVoid vPause  (SourceID u8PossibleNextSource, tU16 u16SinkID);
   tVoid vWaitFor(tU16 u16SinkID);

   /**
    * save the given data as userdata in a member
    */
   tVoid vSetUserData(tU32 u32UserData);

   /**
    * returns the data stored as userdata in a member
    */
   tU32 u32GetUserData(){return m_u32UserData;}

   tVoid vSetCurActSink(tU16 u16SinkID);
   tU8 vSetValidSink(tU16 u16SinkID);
   tU16 u16GetCurActSink();
   tVoid RestoreSource(std::string Src);

   /**
    * returns the audio source index stored as possible next source in a member
    */
   SourceID u8GetNextSource(){return m_possibleNextSource;}
   tBool bIsActive();
   tBool bIsActive(tU16 u16SinkID);

   tBool bIsOff();
   tBool bIsOff(tU16 u16SinkID);
   virtual tVoid vReset();

   virtual tVoid vNotifyStartUnMute(tU16 SinkID);
   virtual tVoid vNotifyStartMute(tU16 SinkID);

   virtual tVoid vNotifyOnDone(tU16 SinkID);
   virtual tVoid vNotifyOffDone(tU16 SinkID); // can be overwritten by the corresponding source to reset the internal states
   virtual tVoid vNotifyPauseDone(tU16 SinkID);
   virtual tVoid vNotifyInitStarted(tU16 SinkID);

   /**
    * returns the string name of this audio source implementation
    */
   virtual const tChar *pacGetName(){return m_pacName;}
   SourceID sGetId();

   const clSourceClass& getSourceClass();
   const stSourceType& getSourceType();
   //const bool bIsSourceRestorable();
   bool bIsSourceRestorable();

   const tChar* pacGetState();
   const tChar* pacGetState(tU16 u16SinkID);

   SourceStateMachine::clSrcState* pclGetState();
   SourceStateMachine::clSrcState* pclGetState(tU16 u16SinkID);

   void pclSetState(SourceStateMachine::clSrcState* pclState);
   void pclSetState(SourceStateMachine::clSrcState* pclState, tU16 u16SinkID);

   void addSink(int sinkID);  //daw2hi 05.09 add sink ID to AudioSource

   virtual tVoid vPrint();

   virtual const am::am_Handle_s* getHandle(const am_Handle_s* p) {(void)p; return (am_Handle_s*)NULL;} //This has no handle;

   virtual const am::am_Handle_s* getSinkHandle(const am_Handle_s* p) {(void)p; return (am_Handle_s*)NULL;} //This has no sink handle;
   virtual am::am_sinkID_t getSinkIDByHandle(const am_Handle_s* pHandle);
   virtual void setSinkHandle(const am_Handle_s* p, tU16 a);

   virtual bool eraseSinkHandle(am_Handle_s /* handle */) {return true;}

   virtual void setMainSourceSoundProperty() {return;}  //This is not using it

protected:
   // Audio Source State Machine
   tVoid Initialize(const tChar *pacName, SourceID enSrcId);
   /**
    * holds the current state of this audio source
    */
   SourceStateMachine::clSrcState* m_pclState;


   // we should have a map ??
   // sink -> state
   std::map<tU16,SourceStateMachine::clSrcState*> sinkStateMap;

   clAudioSource(const tChar *pacName, SourceID enSrcId);

   /**
    * holds the string name of this audio source
    */
  const tChar *m_pacName;

   /**
    * holds the id of this audio source
    */
  SourceID m_SrcId;

   /**
    * holds the possible next source set by the member function
    */
   SourceID m_possibleNextSource;
   /**
    * Note! m_u32UserData is currently used for preset-data only. If you need it for
    * further data, you will have to apply a bit-mask to separate the data!
    */
   tU32 m_u32UserData;

   // ToDo later we will keep a vector of state machines and a vector of sinks
   tU16 m_curActSinkID;
   std::vector<tU16> ValidSink;



public:
   // internal IF for MW CCA Communication
   virtual tVoid vMW_On(tU16 SinkID);
   virtual tVoid vMW_Off(SourceID possibleNextSource,tU16 SinkID);
   virtual tVoid vMW_Pause(SourceID possibleNextSource,tU16 SinkID);
   virtual tVoid vMW_Init(tU16 SinkID);

   virtual tVoid vMW_StartMute(const am_Handle_s* pHandle,tU16 SinkID);
   virtual tVoid vMW_StartUnMute(const am_Handle_s* pHandle,tU16 SinkID);

   //internal IF to ensure that the source is ready for activation
   virtual enSourceAvailability enIsSourceAvailable();

   // check play cmd counter to allow/disallow stop command
   tBool bPreCheckStateChange2Off();
   inline tVoid vResetPlayCmdCounter(){m_u32PlayCmdCounter = 0;};

   /**
    * informs the source whether it is the LSM source
    */
   tBool bIsSourceLSM();
   tVoid vSetSourceLSM(tBool bLSM);

   tVoid vSetSourceBG(tBool bBGsrc);
   tVoid vSetPrevSrcRemoved(tBool bPrevSrcRemoved);
   tBool bIsSourceBG();
   tBool bIsPrevSourceRemoved();

   tVoid vSrcChgUnAvailabilityReason();

   //still needed ?
  // tVoid vMW_Muted();
   tVoid vMW_Muted(const am_Handle_s* pHandle,tU16 SinkID=1);

   tVoid vMW_UnMuted();
   virtual tVoid vMW_UnMuted(const am_Handle_s* pHandle,tU16 SinkID=1);

protected:
   // internal IF for MW CCA Communication


   tVoid vMW_Muted(tU16 u16SinkID);


   tVoid vMW_UnMuted(tU16 u16SinkID);

   tVoid vMW_OnDone();
   tVoid vMW_OnDone(tU16 u16SinkID);

   tVoid vMW_OffDone();
   tVoid vMW_OffDone(tU16 u16SinkID);

   tVoid vMW_PauseDone();
   tVoid vMW_PauseDone(tU16 u16SinkID);

   // to be called when a source finished to prepare and a play to FC-Audio is possible
   tVoid vMW_InitDone();
   tVoid vMW_InitDone(tU16 u16SinkID);

   // to be called when a source received a Confirmation from FC-Audio for a Play-Request
   // this is required to differentiate between the states: rampUpRequesting and rampUp
   // during RampUpRequesting, a new stop-request has to be confirmed manually, as FC-Audio will
   // not confirm.
   tVoid vMW_RequestConfirmed();
   tVoid vMW_RequestConfirmed(tU16 u16SinkID);

   // to be called when a required mw-service for a source becomes available
   // this is required to re-sync the source (finish the rampDown or restart)
   tVoid vMW_ServiceAvailable();
   tVoid vMW_ServiceAvailable(tU16 u16SinkID);

   // to be called when a required mw-service for a source becomes unavailable
   // currently not required. for future use only.
   tVoid vMW_ServiceUnavailable();
   tVoid vMW_ServiceUnavailable(tU16 u16SinkID);

   /**
    * starts timer that will set source to unavailable if it is
    * in unkown state
    */
   tVoid vWatchdogStart_Availability();

   //**************************
   //****      MEMBERS
   //**************************
   tU32 m_u32PlayCmdCounter;
   /**
    * m_StackRules holds the rules for the order of sources on the stack
    */
   //clStackRules m_StackRule;
   /*
    * Typically AudioSOurce Controller
    */
   clAudioSourceObserver* m_pclAudioSourceObserver;
   /**
    * actual availability state
    */
   enSourceAvailability m_availability;

   /**
    * reason for availability
    */
   enSourceAvailabilityReason m_availabilityReason;

   /**
    * member that is set when source is LSM during startup
    */
   tBool m_bSourceIsLSM;

   /**
   * member that is set when source is background source bening exchanged
   **/
   tBool m_bBackgroundSrcExchange;

   /**
   * member that is set when previous source is removed either due to device removed from system OR
   * availability message not received on startup
   **/
   tBool m_bPrevSrcRemoved;

private:
  static CAmSocketHandler*    m_pSocketHandler;
  am::sh_timerHandle_t m_handle;//Handle for the timer

  bool bStartTimer(uint32_t seconds);//Helper function to start the timer

public:
  //Callback from mainloop, to inform that the timer has expired
  virtual void Call(const am::sh_timerHandle_t handle, void* userData);
};

}}//namespace

#endif // _AUDIOSOURCE_H_
