//
// TypeConversions/Functions.h
//
// provide conversion functions from
//     - FI-types to internal types
//     - Amplifier-types to internal types
// and vice versa.
//
//  Created on: Jun 25, 2014
//      Author: Martin Koch, Fa. ESE
//

// requires prior loading of
//   - midw_fi_if.h
//   - ADR3StreamType.h



#ifndef TYPECONVERSIONFUNCTIONSS_H_
#define TYPECONVERSIONFUNCTIONSS_H_



namespace VolumeManager
{

   // ==========================================================================
   //
   //              generic  C o n v e r s i o n - M a p
   //
   // this class is defined as template, providing a unified interface to
   // all conversion maps.
   //
   // Mapping tables and defaults for dedicated conversions
   // to be provided in MappingData.hpp
   //
   // --------------------------------------------------------------------------

   template<typename T1, typename T2>
   /* static */ class ConversionMap
   {
   private:
      struct Data
      {
         T1 first;
         T2 second;
      };

      // member variables
      static const Data data[];
      static const size_t size;

      /* default constructor */ ConversionMap();  // never instantiate this class

   public:
      // lookup functions
      static T1 getFirst(T2 second);
      static T2 getSecond(T1 first);

   };

   // =============================================================================
   //
   //                          I m p l e m e n t a t i o n
   //

   template<typename T1, typename T2>
   /* static */ T1 ConversionMap <T1, T2> :: getFirst(T2 second)
   {
      for (size_t i = 1; i < size; ++i)
         if (data[i].second == second)
            return data[i].first;

      // default
      return data[0].first;
   }

   // --------------------------------------------------------------------------

   template<typename T1, typename T2>
   /* static */ T2 ConversionMap <T1, T2> :: getSecond(T1 first)
   {
      for (size_t i = 1; i < size; ++i)
         if (data[i].first == first)
            return data[i].second;

      // default
      return data[0].second;
   }

   // ==========================================================================
   //
   //              special implemented  C o n v e r s i o n - M a p s
   //
   // provide shorthand names for all implemented conversion maps.
   //

#ifdef MIDW_FI_S_ALREADY_INCLUDE_FI_TYPES  // only offer these if mascf_fi types are available
   // -----------------------------------------------------------------------------
   //
   //                   FI-AudioChannel  <-->  internal StreamID

   typedef ConversionMap <midw_fi_tcl_e8_AudioChannel::tenType, tenStream>   fiAudioChannel_StreamID_Map;

   // -----------------------------------------------------------------------------
   //
   //                   FI - volume mode  <-->  internal volume mode

   typedef ConversionMap<midw_fi_tcl_e8_VolumeMode::tenType, tenVolumeMode>  fiVolumeMode_Map;

   // -----------------------------------------------------------------------------
   //
   //                   FI - resource  <-->  internal resource

   typedef ConversionMap<midw_fi_tcl_e8_ResourceNo::tenType, tenResource>  fiResource_Map;

   // -----------------------------------------------------------------------------
   //
   //                   FI - resource  <-->  internal resource

   typedef ConversionMap<midw_fi_tcl_e8_AudSource::tenType, tenInternalSource>  fiSource_Map;

   // -----------------------------------------------------------------------------
   //
   //                       FI - volume change direction

   typedef ConversionMap<midw_fi_tcl_e8_AudioVolumeType::tenType, tenValueType>  fiValueType_Map;

   // -----------------------------------------------------------------------------
   //
   //                       FI - volume list entry

   typedef ConversionMap<midw_fi_tcl_e8_Aud_VolumeType::tenType, tU8>   fiVolumeListEntry_Map;

   // -----------------------------------------------------------------------------
   //
   //                       FI - volume lock state

   typedef ConversionMap<midw_fi_tcl_e8_VolumeLock::tenType, tenVolumeLock>   fiVolumeLockState_Map;

   // -----------------------------------------------------------------------------
   //
   //                       FI - volume requester state

   typedef ConversionMap<midw_fi_tcl_e8_Requester::tenType, tenVolumeLockRequester>   fiVolumeLockRequester_Map;

   // -----------------------------------------------------------------------------
   //
   //                       FI - amplifier thermal warning level

   typedef ConversionMap<midw_fi_tcl_e8_AmpTemperatureLevel::tenType, tenErrorValue>   fiAmplifierWarningLevel_Map;

   // --------------------------------------------------------------------------

#endif  // MIDW_FI_S_ALREADY_INCLUDE_FI_TYPES

}   // namespace VolumeManager

#endif /* TYPECONVERSIONFUNCTIONSS_H_ */
