//
// VolumeManager/dBCalculator.cpp
//
//  calculate dB-values from user-step values for volume
//   use configurable lookup table
//
//  Created on: Jul 7, 2014
//      Author: Martin Koch, Fa. ESE
//



//#define ETRACE_S_IMPORT_INTERFACE_GENERIC
//#include <etrace_if.h> // implicitly links <osal_if.h>

#include "Volume/Types.h"
#include "Volume/Utilities/Uncopyable.h"
#include "./dBCalculator.h"
// - - - - - - - - - - - - - -

#include "Volume/Configuration/XMLParser.h"
#include "Volume/Utilities/StringCopy.hpp"

#include "controllerplugin_Trace.h"
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_CONTROLLER_VOLUME
#include "trcGenProj/Header/dBCalculator.cpp.trc.h"



namespace VolumeManager
{

#define UNDEF_VALUE 255
#define MAX_TRIAL 5
   // --------------------------------------------------------------------------
   //
   //    file-locals

   char defaultCalulatorName[] = "intrinsic default calculator";
   tDBTable defaultData[]
    = {
          {0, -120, -120}
        , {1,  -80,  -80}
        , {2,  -70,  -70}
        , {3,  -65,  -65}
        , {4,  -60,  -60}
        , {5,  -56,  -56}
        , {6,  -52,  -52}
        , {7,  -48,  -48}
        , {8,  -44,  -44}
        , {9,  -40,  -40}
        , {10, -37,  -37}
        , {11, -34,  -34}
        , {12, -31,  -31}
        , {13, -28,  -28}
        , {14, -25,  -25}
        , {15, -22,  -22}
        , {16, -20,  -20}
        , {17, -18,  -18}
        , {18, -16,  -16}
        , {19, -14,  -14}
        , {20, -12,  -12}
        , {21, -10,  -10}
        , {22,  -8,   -8}
        , {23,  -7,   -7}
        , {24,  -6,   -6}
        , {25,  -5,   -5}
        , {26,  -4,   -4}
        , {27,  -3,   -3}
        , {28,  -2,   -2}
        , {29,  -1,   -1}
        , {30,   0,    0}
      };

   // --------------------------------------------------------------------------

   template<typename T>
   inline void swapMember (T& lhs, T& rhs)
   {
      T tmp = lhs;
      lhs = rhs;
      rhs = tmp;
   };

   // --------------------------------------------------------------------------

   // ADR3 is using 1/4 dB values, thus the defines of the system feature list need to be multiplied
   /* static */ const tS16 dBCalculator:: dBStepMultiplicator = 1;

   // --------------------------------------------------------------------------

   /* constructor */ dBCalculator:: dBCalculator (tDBTable* pNewTable, unsigned u32TableRowCount, const char* name, tenStream enSecondaryStream, tenResource sink)
      : _pDataFromXML(pNewTable)
      , _pData(pNewTable)
      , _size(u32TableRowCount)
      , _name(NULL)
      , _secondaryStreamID(enSecondaryStream)
      , _sink(sink)
   {
      const char unnamed[] = "unnamed";
      if (pNewTable && u32TableRowCount)
      {
         _name = stringCopy((name ? name : unnamed));

         ETG_TRACE_USR2(("dBCalculator() initialized with %u rows for '%s'", _size, _name))
      }
      else
      {
         // fallback to intrinsic defaults
         _pData = defaultData;
         _size = sizeof(defaultData) / sizeof(tDBTable);
         _name = stringCopy(defaultCalulatorName);

         ETG_TRACE_FATAL(("E R R O R :  failed to initialize dBCalculator '%s'", (name ? name : unnamed)))
         ETG_TRACE_USR2(("             using '%s' instead", defaultCalulatorName))
      }
   }

   // --------------------------------------------------------------------------

   /* default constructor */ dBCalculator:: dBCalculator ()
      : _pDataFromXML(NULL)
      , _pData(NULL)
      , _size(0)
      , _name(NULL)
      , _secondaryStreamID(EN_AUDIO_SOURCE_STREAM_DEFAULT)
      , _sink(UndefinedResource)
   {

   }

   // --------------------------------------------------------------------------

   /* destructor */ dBCalculator:: ~dBCalculator ()
   {
      delete _pDataFromXML;
      _pDataFromXML = NULL;
      _pData = NULL;

      delete _name;
      _name = NULL;
   }

   // --------------------------------------------------------------------------

   void dBCalculator:: swap (dBCalculator& rhs)
   {
      swapMember<tDBTable*>(_pDataFromXML, rhs._pDataFromXML);
      swapMember<tDBTable*>(_pData, rhs._pData);
      swapMember<size_t>(_size, rhs._size);
      swapMember<char*>(_name, rhs._name);
      swapMember<tenStream>(_secondaryStreamID, rhs._secondaryStreamID);
      swapMember<tenResource>(_sink, rhs._sink);
   }

   // --------------------------------------------------------------------------

   tS16 dBCalculator:: s16GetdBValue (tU8 u8UserStep, tenStream enStream) const
   {
      if (NULL == _pData)
         return -120;   // better inaudible than roaring

      if (u8GetMaximumUserStep() < u8UserStep)
      {  // should not happen
         ETG_TRACE_FATAL(("dBCalulator() - E R R O R:  VolumeStep out of Range: %u for curve %s"
            , u8UserStep, _name))
         u8UserStep = u8GetMaximumUserStep();
      }

      tS16 retVal;
      if ((_secondaryStreamID != EN_AUDIO_SOURCE_STREAM_DEFAULT) && (_secondaryStreamID == enStream))
         retVal = _pData[u8UserStep].s16Secondary;
      else
         retVal = _pData[u8UserStep].s16Primary;

      ETG_TRACE_USR3(("dBCalculator returning %d dB for user-step %u from curve %s"
         , retVal, u8UserStep, _name))
      return retVal;
   }

   // --------------------------------------------------------------------------

   tU8  dBCalculator:: u8GetMinimumUserStep () const
   {
      if (NULL == _pData)
         return 0;

      return _pData[0].u8UserStep;
   }

   // --------------------------------------------------------------------------

   tU8  dBCalculator:: u8GetMaximumUserStep () const
   {
      if (NULL == _pData)
         return 0;

      return _pData[_size - 1].u8UserStep;
   }

   // --------------------------------------------------------------------------

   const char* dBCalculator:: sGetName() const
   {
      return _name;
   }

   // --------------------------------------------------------------------------

   tenStream dBCalculator:: enGetSecondaryStreamID() const
   {
      return _secondaryStreamID;
   }
   // --------------------------------------------------------------------------
   tenResource dBCalculator:: enGetSinkID() const
   {
      return _sink;
   }
   // --------------------------------------------------------------------------
   tU8  dBCalculator::u8GetStepValue(tS16 dbValue) const
   {
        if (NULL == _pData)
           return 0;   // better inaudible than roaring

        tU8 retVal = UNDEF_VALUE;
        tU8 i = 0;
        tS16 IntCpyDBVal = dbValue;

        //Max check for 5 trials for closest Step value of corresponding db
        for(; i < MAX_TRIAL; i++)
        {
          retVal = u8Calculate(static_cast<tS16>(IntCpyDBVal + i));

          if(retVal != UNDEF_VALUE)
          {
            ETG_TRACE_USR3(("u8GetStepValue returned : %d",  retVal))
            return retVal;
          }
          else
          {
             retVal = u8Calculate(static_cast<tS16>(IntCpyDBVal - i));

            if(retVal != UNDEF_VALUE)
            {
              ETG_TRACE_USR3(("u8GetStepValue returned : %d",  retVal))
              return retVal;
            }
          }
        }
        return retVal; // PC Please don't come here..
   }

   // --------------------------------------------------------------------------
   tU8 dBCalculator::u8Calculate(tS16 dbValue) const
   {
     tU8 retVal = UNDEF_VALUE;
     tU8 iterVal = u8GetMinimumUserStep(); //Start from minVolStep

     if (NULL == _pData)
        return 0;   // better inaudible than roaring

     for(; iterVal <= u8GetMaximumUserStep(); iterVal++ )
     {
         if( dbValue == _pData[iterVal].s16Primary)
         {
           retVal = iterVal;
         }
     }

     return retVal;
   }
   // --------------------------------------------------------------------------
}   // namespace VolumeManager

