//
// VolumeManager/Configuration/Configuration.cpp
//
// Provide configuration details for audio sources regarding
//    source grouping
//    persistence
//    min/max/default user-step constraints
//    related output stream,
//    calculation of dB-values forwarded to amplifier
// as defined in file "Configuration.xml"
//
//  Created on: Aug 1, 2014
//      Author: Martin Koch, Fa. ESE
//




#include "Volume/Types.h"
#include "./Configuration.h"
// - - - - - - - - - - - -


#include "./ConfigDetails.hpp"
#include "./XMLParser.h"
#include "./dBCalculator.h"
#include "Volume/Utilities/StringCopy.hpp"



#include "controllerplugin_Trace.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CONTROLLER_VOLUME
#include "trcGenProj/Header/Configuration.cpp.trc.h"
#define FI_EN_VOLUME_TYPE_NOT_DEF 255




namespace VolumeManager
{

   // --------------------------------------------------------------------------

   /* constructor */ Configuration::Configuration (const char* ConfigFile)
      : _pCalculators(NULL)
      , _pFunctions(NULL)
      , _pSets(NULL)
      , _pStreams(NULL)
      , _pStreamMap(NULL)
      , _pGroups(NULL)
      , _pSources(NULL)
      , _pConfigFile(ConfigFile)
      , _calculatorCount(0)
      , _functionCount(0)
      , _setCount(0)
      , _streamCount(0)
      , _sourceCount(0)
      , _sourceGroupCount(0)

   {

//    ET_TRACE_OPEN;

      XMLParser parser(_pConfigFile);
      parser.vPopulateCalculators(/* output */ _pCalculators, _calculatorCount);

      parser.vPopulateFunctions(/* output */ _pFunctions, _functionCount);

      parser.vPopulateStreamSets(_pFunctions, _functionCount
            , /* output */ _pSets, _setCount, _pStreams,  _streamCount, _pCalculators, _calculatorCount);

      parser.vPopulateSourceGroups(_pStreams, _streamCount, _pCalculators, _calculatorCount
            , /* output */ _pStreamMap, _pGroups, _sourceGroupCount);

      parser.vPopulateSources(_pGroups, _sourceGroupCount, /* output */ _pSources, _sourceCount);

      for(size_t i= 0; i < _sourceCount; ++i )
      {
          std::string sourceName(_pSources[i].name);
          sourceList.insert(sourceName);
      }

      for(size_t i= 0; i < _setCount; ++i )
      {
          std::string sinkName(_pSets[i].name);
          sinkList.insert(sinkName);
      }





   }

   // --------------------------------------------------------------------------

   /* destructor */ Configuration:: ~Configuration ()
   {
      if (_pSources)
         for (size_t i = 0; i < _sourceCount; ++i)
            delete _pSources[i].name;
      delete[] _pSources;

      if (_pGroups)
         for (size_t i = 0; i < _sourceGroupCount; ++i)
            delete _pGroups[i].name;
      delete[] _pGroups;

      delete[] _pStreamMap;

      if (_pStreams)
         for (size_t i = 0; i < _streamCount; ++i)
           delete _pStreams[i].name;
      delete[] _pStreams;

      if (_pSets)
         for (size_t i = 0; i < _setCount; ++i)
            delete _pSets[i].name;
      delete[] _pSets;

      if (_pFunctions)
         for (size_t i = 0; i < _functionCount; ++i)
         {
            delete _pFunctions[i].name;
            delete _pFunctions[i].params;
            delete _pFunctions[i].groupRefs;
         }
      delete[] _pFunctions;

      delete[] _pCalculators;


      ETG_TRACE_USR1(("Configuration dissolved."))
   }

   // --------------------------------------------------------------------------

   size_t Configuration:: getStreamSetCount () const
   {
      return _setCount;
   }

   // --------------------------------------------------------------------------

   size_t Configuration:: getStreamCount () const
   {
      return _streamCount;
   }

   // --------------------------------------------------------------------------

   size_t Configuration:: getSourceGroupCount () const
   {
      return _sourceGroupCount;
   }

   // --------------------------------------------------------------------------

   const VolumeConstraints* Configuration:: pGetVolumeLimitsByGroup (tU8 u8GroupTypeId, bool requestPersistenceLimits) const
   {
      for (size_t i = 0; i < _sourceGroupCount;  ++i)
      {
         if (_pGroups[i].typeId == u8GroupTypeId)
         {
            if (requestPersistenceLimits)
               return &(_pGroups[i].persistenceConstraints);
            else
               return &(_pGroups[i].constraints);
         }
      }
      // not found
      return NULL;
   }

   // --------------------------------------------------------------------------

   const StreamSetConfig* Configuration:: pGetStreamSetConfig (size_t configIndex) const
   {
      if (configIndex < _setCount)
         return &_pSets[configIndex];
      else
         return NULL;
   }

   // --------------------------------------------------------------------------

   const StreamConfig* Configuration:: pGetStreamConfig (size_t configIndex) const
   {
      if (configIndex < _streamCount)
         return &_pStreams[configIndex];
      else
         return NULL;
   }

   // --------------------------------------------------------------------------

   const dBCalculator* Configuration:: pGetDBCalculator (tenInternalSource enSource) const
   {
      if (_pSources)
         for (size_t i = 0; i < _sourceCount; ++i)
            if (enSource == _pSources[i].enGetSourceID())
            {
               const dBCalculator* pCalc = _pSources[i].pGetCalculator();
               if (pCalc)
                  return pCalc;
               else
                  break;  // not valid
            }

      // not found or not valid
      ETG_TRACE_FATAL(("getDBCalculator() - E R R O R :  could not locate dBCalculator for source %u"
            , ETG_CENUM(tenInternalSource, enSource)));

      return NULL;
   }

   // --------------------------------------------------------------------------

   const SourceConfig* Configuration:: pGetSourceConfig (tenInternalSource enSource) const
   {
      if (_pSources)
      {
         for (size_t i = 0; i < _sourceCount; ++i)
         {
             if (enSource == _pSources[i].enGetSourceID())
            {
              ETG_TRACE_USR4(("pGetSourceConfig  return &_pSources[%d] with ID %d and name %s",i,enSource,_pSources[i].sGetName()));
               return &_pSources[i];
            }
         }
      }
      ETG_TRACE_USR2(("pGetSourceConfig() - E R R O R : could not locate config for source %u"
            , ETG_CENUM(tenInternalSource, enSource)))
      return NULL;
   }

   // --------------------------------------------------------------------------

   void Configuration:: vGetSourceGroupConfigs (const GroupConfig*& ext_pGroups, size_t& ext_NumberOfGroups) const
   {
      ext_pGroups = _pGroups;
      ext_NumberOfGroups = _sourceGroupCount;
   }

   // --------------------------------------------------------------------------

    const StreamSetConfig* Configuration::pGetStreamSetConfig(const std::string& streamSetName) const
    {
        for(size_t configIndex = 0;configIndex < _setCount; ++configIndex)
        {
           if(_pSets[configIndex].name == NULL) continue;
           std::string name(_pSets[configIndex].name);
           if(name ==  streamSetName)
               return &_pSets[configIndex];
        }
           return NULL;

    }

    const SourceConfig* Configuration::pGetSourceConfig(const std::string& sourceName) const
    {
        if (_pSources)
         {
            for (size_t i = 0; i < _sourceCount; ++i)
            {
               if(_pSources[i].sGetName() == NULL) continue;
               std::string name(_pSources[i].sGetName());
               if (name == sourceName)
               {
                 ETG_TRACE_USR4(("pGetSourceConfig  return &_pSources[%d] with ID %d and name %s",i,_pSources[i].enGetSourceID(),_pSources[i].sGetName()));
                  return &_pSources[i];
               }
            }
         }

         return NULL;
    }

    const dBCalculator* Configuration::pGetDBCalculator(const std::string& sourceName) const
    {
        if (_pSources)
        {
           for (size_t i = 0; i < _sourceCount; ++i)
           {
              if(_pSources[i].sGetName() == NULL) continue;
              std::string name(_pSources[i].sGetName());
              if (name == sourceName)
              {
                 const dBCalculator* pCalc = _pSources[i].pGetCalculator();
                 if (pCalc)
                    return pCalc;
                 else
                    break;  // not valid
              }

           }

        // not found or not valid
        ETG_TRACE_FATAL(("getDBCalculator() - E R R O R :  could not locate dBCalculator for source %s"
              , sourceName.c_str() ));
        }

        return NULL;
     }

    bool Configuration::validateSink(const std::string& sinkName) const
    {
        auto it = sinkList.find(sinkName);
        if(it != sinkList.end())
            return true;
        return false;

    }

    bool Configuration::validateSource(const std::string& sourceName) const
    {
        auto it = sourceList.find(sourceName);
        if(it != sourceList.end())
            return true;
        return false;
    }

    std::string Configuration::getGroupName(const std::string& sourceName) const
    {
        const SourceConfig* p = pGetSourceConfig(sourceName);
        if(p == NULL) return "NONE";
        std::string s = p->getGroupName();
        //delete p;
        return s;

    }

    unsigned char Configuration::getGroupTypeId(const std::string& sourceName) const
    {
        const SourceConfig* p = pGetSourceConfig(sourceName);
        if(p == NULL) return 255; // FI_EN_VOLUME_TYPE_NOT_DEF
        return p->u8GetGroupType();

    }

    unsigned char Configuration:: getEntertainmentGroupID (void) const
    {
    for (size_t i = 0; i < _sourceGroupCount;  ++i)
    {
        if(_pGroups[i].name != NULL)
        {
         std::string name(_pGroups[i].name);
         if (name == "Entertain")
          {
              ETG_TRACE_USR3(("Entertainment Group ID  =%d ",_pGroups[i].typeId));
              return (_pGroups[i].typeId);
          }

        }
    }
       // not found
       ETG_TRACE_USR3(("Returning FI_EN_VOLUME_TYPE_NOT_DEF"));
       return FI_EN_VOLUME_TYPE_NOT_DEF; // FI_EN_VOLUME_TYPE_NOT_DEF = 255
    }

}
